IMPLEMENTATION MODULE LowLevel;

	(********************************************************)
	(*							*)
	(*	   Miscellaneous low-level procedures		*)
	(*							*)
	(*  Programmer:		P. Moylan			*)
	(*  Last edited:	9 February 1994			*)
	(*  Status:		Apparently working, but not	*)
	(*			tested extensively		*)
	(*							*)
	(*	Note that the implementation of this module	*)
	(*	is heavily compiler-dependent.  This version	*)
	(*	is a "semi-portable" version, to use where	*)
	(*	a compiler other than the TopSpeed compiler	*)
	(*	is used.  It will almost certainly need some	*)
	(*	tailoring for each separate compiler.		*)
	(*							*)
	(********************************************************)

IMPORT SYSTEM, Lib;

TYPE
    t02 = [0..2];

    Table = ARRAY [0..15] OF CARDINAL;

    Word =  RECORD
		CASE :t02 OF
		    0:	bits: BITSET;
		  |
		    1:	low, high: BYTE;
		  |
		    2:	w: CARDINAL;
		END (*CASE*);
	    END (*RECORD*);

    Double = RECORD
		CASE :t02 OF
		    0:	low, high: CARDINAL;
		  |
		    1:	lw: LONGCARD;
		  |
		    2:	a: ADDRESS;
		END (*CASE*);
	     END (*RECORD*);

CONST
    power2 = Table (1, 2, 4, 8, 16, 32, 64, 128, 256, 512,
			1024, 2048, 4096, 8192, 16384, 32768);

(************************************************************************)
(*			    BITWISE LOGIC				*)
(************************************************************************)

PROCEDURE IAND (first, second: WORD): CARDINAL;

    (* Bit-by-bit logical AND.	*)

    VAR a, b, result: Word;

    BEGIN
	a.w := first;  b.w := second;
	result.bits := a.bits * b.bits;
	RETURN result.w;
    END IAND;

(************************************************************************)

PROCEDURE IANDB (first, second: BYTE): SHORTCARD;

    (* Bit-by-bit logical AND for bytes.	*)

    BEGIN
	RETURN VAL(BYTE, IAND(VAL(WORD,first), VAL(WORD, second)));
    END IANDB;

(************************************************************************)

PROCEDURE IOR (first, second: WORD): CARDINAL;

    (* Bit-by-bit inclusive OR.	*)

    VAR a, b, result: Word;

    BEGIN
	a.w := first;  b.w := second;
	result.bits := a.bits + b.bits;
	RETURN result.w;
    END IOR;

(************************************************************************)

PROCEDURE IORB (first, second: BYTE): SHORTCARD;

    (* Bit-by-bit inclusive OR.	*)

    BEGIN
	RETURN VAL(BYTE, IOR(VAL(WORD,first), VAL(WORD, second)));
    END IORB;

(************************************************************************)

PROCEDURE IXOR (first, second: WORD): CARDINAL;

    (* Bit-by-bit exclusive OR.	*)

    VAR a, b, result: Word;

    BEGIN
	a.w := first;  b.w := second;
	result.bits := a.bits / b.bits;
	RETURN result.w;
    END IXOR;

(************************************************************************)

PROCEDURE IXORB (first, second: BYTE): SHORTCARD;

    (* Bit-by-bit exclusive OR.	*)

    BEGIN
	RETURN VAL(BYTE, IXOR(VAL(WORD,first), VAL(WORD, second)));
    END IXORB;

(************************************************************************)

PROCEDURE INOT (value: WORD): CARDINAL;

    (* Bit-by-bit Boolean complement.	*)

    BEGIN
	RETURN 0FFFFH-VAL(CARDINAL,value);
    END INOT;

(************************************************************************)

PROCEDURE INOTB (value: BYTE): SHORTCARD;

    (* Bit-by-bit Boolean complement.	*)

    BEGIN
	RETURN 0FFH-VAL(SHORTCARD,value);
    END INOTB;

(************************************************************************)

PROCEDURE ROL (value: WORD;  count: CARDINAL): CARDINAL;

    (* Left rotation of "value" by "count" bit positions.	*)

    BEGIN
	count := count MOD 16;
	RETURN LS(value, count) + RS(value, 16-count);
    END ROL;

(************************************************************************)

PROCEDURE ROLB (value: BYTE;  count: CARDINAL): SHORTCARD;

    (* Left rotation of "value" by "count" bit positions.	*)

    BEGIN
	count := count MOD 8;
	RETURN LSB(value, count) + RSB(value, 8-count);
    END ROLB;

(************************************************************************)

PROCEDURE LS (value: WORD;  count: CARDINAL): CARDINAL;

    (* Left shift of "value" by "count" bit positions, with zero fill.	*)

    BEGIN
	IF count > 15 THEN RETURN 0
	ELSIF count = 0 THEN RETURN value
	ELSE
	    value := IAND (value, power2[16-count]-1);
	    RETURN VAL(CARDINAL,value) * power2[count];
	END (*IF*);
    END LS;

(************************************************************************)

PROCEDURE LSB (value: BYTE;  count: CARDINAL): SHORTCARD;

    (* Left shift of "value" by "count" bit positions, with zero fill.	*)

    BEGIN
	RETURN LowByte (LS(VAL(WORD,value), count));
    END LSB;

(************************************************************************)

PROCEDURE ROR (value: WORD;  count: CARDINAL): CARDINAL;

    (* Right rotation of "value" by "count" bit positions.	*)

    BEGIN
	count := count MOD 16;
	RETURN RS(value, count) + LS(value, 16-count);
    END ROR;

(************************************************************************)

PROCEDURE RORB (value: BYTE;  count: CARDINAL): SHORTCARD;

    (* Right rotation of "value" by "count" bit positions.	*)

    BEGIN
	count := count MOD 8;
	RETURN RSB(value, count) + LSB(value, 8-count);
    END RORB;

(************************************************************************)

PROCEDURE RS (value: WORD;  count: CARDINAL): CARDINAL;

    (* Right shift of "value" by "count" bit positions, with zero fill.	*)

    BEGIN
	(*# save, check(overflow => off) *)
	IF count > 15 THEN RETURN 0
	ELSE RETURN VAL(CARDINAL,value) DIV power2[count];
	END (*IF*);
	(*# restore *)
    END RS;

(************************************************************************)

PROCEDURE RSB (value: BYTE;  count: CARDINAL): SHORTCARD;

    (* Right shift of "value" by "count" bit positions, with zero fill.	*)

    BEGIN
	RETURN LowByte (RS(VAL(WORD,value), count));
    END RSB;

(************************************************************************)
(*			    POINTER OPERATIONS				*)
(************************************************************************)

PROCEDURE MakePointer (segment, offset: CARDINAL): ADDRESS;

    (* Creates a pointer, given the segment and offset within segment.	*)

    VAR value: Double;

    BEGIN
	value.low := offset;  value.high := segment;
	RETURN value.a;
    END MakePointer;

(************************************************************************)

PROCEDURE SEGMENT (A: ADDRESS): CARDINAL;

    (* Returns the segment part of an address.	*)

    VAR value: Double;

    BEGIN
	value.a := A;
	RETURN value.high;
    END SEGMENT;

(************************************************************************)

PROCEDURE OFFSET (A: ADDRESS): CARDINAL;

    (* Returns the offset part of an address.	*)

    VAR value: Double;

    BEGIN
	value.a := A;
	RETURN value.low;
    END OFFSET;

(************************************************************************)

PROCEDURE Virtual (PA: LONGCARD): ADDRESS;

    (* Converts a physical address to a virtual address, if possible.	*)
    (* There are no guarantees in the case where there is no such	*)
    (* virtual address.							*)

    VAR value: Double;

    BEGIN
	value.low := VAL(CARDINAL,PA MOD 16);
	value.high := VAL(CARDINAL,PA DIV 16);
	RETURN value.a;
    END Virtual;

(************************************************************************)

PROCEDURE Physical (A: ADDRESS): LONGCARD;

    (* Converts a virtual address to a physical address.  Use with care!*)

    VAR value: Double;

    BEGIN
	value.a := A;
	RETURN 16*VAL(LONGCARD,value.high) + VAL(LONGCARD,value.low);
    END Physical;

(************************************************************************)

PROCEDURE AddOffset (A: ADDRESS;  increment: CARDINAL): ADDRESS;

    (* Returns a pointer to the memory location whose physical address	*)
    (* is Physical(A)+increment.  In the present version, it is assumed	*)
    (* that the caller will never try to run off the end of a segment.	*)

    BEGIN
	RETURN Lib.AddAddr (A, increment);
    END AddOffset;

(************************************************************************)

PROCEDURE SubtractOffset (A: ADDRESS;  decrement: CARDINAL): ADDRESS;

    (* Like AddOffset, except that we go backwards in memory.  Running	*)
    (* off the beginning of the segment is an undetected error.		*)

    BEGIN
	RETURN Lib.SubAddr (A, decrement);
    END SubtractOffset;

(************************************************************************)
(*			BYTE/WORD/LONGWORD CONVERSIONS			*)
(************************************************************************)

PROCEDURE LowByte (w: WORD): BYTE;

    (* Returns the low-order byte of its argument.	*)

    VAR value: Word;

    BEGIN
	value.w := w;
	RETURN value.low;
    END LowByte;

(************************************************************************)

PROCEDURE HighByte (w: WORD): BYTE;

    (* Returns the high-order byte of its argument.	*)

    VAR value: Word;

    BEGIN
	value.w := w;
	RETURN value.high;
    END HighByte;

(************************************************************************)

PROCEDURE MakeWord (high, low: BYTE): WORD;

    (* Combines two bytes into a word.  The first argument becomes the	*)
    (* most significant byte of the result.				*)

    VAR value: Word;

    BEGIN
	value.low := low;
	value.high := high;
	RETURN value.w;
    END MakeWord;

(************************************************************************)

PROCEDURE LowWord (w: LONGWORD): WORD;

    (* Returns the low-order word of its argument.	*)

    VAR value: Double;

    BEGIN
	value.lw := w;
	RETURN value.low;
    END LowWord;

(************************************************************************)

PROCEDURE HighWord (w: LONGWORD): CARDINAL;

    (* Returns the high-order word of its argument.	*)

    VAR value: Double;

    BEGIN
	value.lw := w;
	RETURN value.high;
    END HighWord;

(************************************************************************)

PROCEDURE MakeLongword (high, low: WORD): LONGCARD;

    (* Combines two words into a longword.  The first argument becomes	*)
    (* the most significant word of the result.				*)

    VAR value: Double;

    BEGIN
	value.low := low;
	value.high := high;
	RETURN value.lw;
    END MakeLongword;

(************************************************************************)
(*			MISCELLANEOUS ARITHMETIC			*)
(************************************************************************)

PROCEDURE INCV (VAR (*INOUT*) dest: CARDINAL;  src: CARDINAL): BOOLEAN;

    (* Computes dest := dest + src, and returns TRUE iff the addition	*)
    (* produced a carry.						*)

    BEGIN
	IF dest > MAX(CARDINAL) - src THEN
	    DEC (dest, MAX(CARDINAL) - src + 1);
	    RETURN TRUE;
	ELSE
	    INC (dest, src);
	    RETURN FALSE;
	END (*IF*);
    END INCV;

(************************************************************************)

PROCEDURE DECV (VAR (*INOUT*) dest: CARDINAL;  src: CARDINAL): BOOLEAN;

    (* Computes dest := dest - src, and returns TRUE iff the		*)
    (* subtraction produced a borrow.					*)

    BEGIN
	IF dest < src THEN
	    INC (dest, MAX(CARDINAL) - src + 1);  RETURN TRUE;
	ELSE
	    DEC (dest, src);  RETURN FALSE;
	END (*IF*);
    END DECV;

(************************************************************************)

PROCEDURE Mul (A, B: CARDINAL): LONGCARD;

    (* Same as A*B, except for the type of the result.  We provide this	*)
    (* as a general-purpose function since this combination of operands	*)
    (* is often precisely what is wanted.				*)

    BEGIN
	RETURN VAL(LONGCARD,A) * VAL(LONGCARD,B);
    END Mul;

(************************************************************************)

PROCEDURE Div (A: LONGCARD;  B: CARDINAL): CARDINAL;

    (* Same as A DIV B, except for the type of A.  We provide this as	*)
    (* a general-purpose function since this combination of operands	*)
    (* is often precisely what is wanted.				*)

    BEGIN
	RETURN VAL(CARDINAL, A DIV VAL(LONGCARD,B));
    END Div;

(************************************************************************)
(*			     BLOCK MOVES				*)
(************************************************************************)

PROCEDURE Copy (source, destination: ADDRESS;  bytecount: CARDINAL);

    (* Copies an array of bytes from the source address to the		*)
    (* destination address.  In the case where the two arrays overlap,	*)
    (* the destination address should be lower in physical memory than	*)
    (* the source address.						*)

    BEGIN
	Lib.Move (source, destination, bytecount);
    END Copy;

(************************************************************************)

PROCEDURE CopyUp (source, destination: ADDRESS;  bytecount: CARDINAL);

    (* A variant of Copy which does the move backwards, in order	*)
    (* to handle the case where the destination address is inside the	*)
    (* source array.  In this special case Copy cannot be used,		*)
    (* because it would overwrite data it was about to copy.		*)

    BEGIN
	Lib.Move (source, destination, bytecount);
    END CopyUp;

(************************************************************************)

PROCEDURE BlockFill (destination: ADDRESS;  bytecount: CARDINAL;  value: BYTE);

    (* Fills the destination array with the given value.	*)

    BEGIN
	Lib.Fill (destination, bytecount, value);
    END BlockFill;

(************************************************************************)

PROCEDURE BlockFillWord (destination: ADDRESS;  wordcount: CARDINAL;
							value: WORD);

    (* Fills the destination array with the given value.	*)

    BEGIN
	Lib.WordFill (destination, wordcount, value);
    END BlockFillWord;

(************************************************************************)
(*			    INPUT AND OUTPUT				*)
(************************************************************************)

PROCEDURE OutByte (port: CARDINAL; value: BYTE);

    (* Puts the value out to an output port.	*)

    BEGIN
	SYSTEM.Out (port, value);
    END OutByte;

(************************************************************************)

PROCEDURE InByte (port: CARDINAL): BYTE;

    (* Reads a byte from an input port.	*)

    BEGIN
	RETURN SYSTEM.In (port);
    END InByte;

(************************************************************************)

PROCEDURE InStringWord (port: CARDINAL;  BufferAddress: ADDRESS;
						count: CARDINAL);

    (* Reads count words from an input port.	*)

    VAR j: CARDINAL;  p: POINTER TO WORD;

    BEGIN
	p := BufferAddress;
	FOR j := 1 TO count DO
	    p^ := SYSTEM.InW (port);
	    Lib.IncAddr (p, 2);
	END (*FOR*);
    END InStringWord;

(************************************************************************)

PROCEDURE OutStringWord (port: CARDINAL;  BufferAddress: ADDRESS;
						count: CARDINAL);

    (* Writes count words to an output port.	*)

    VAR j: CARDINAL;  p: POINTER TO WORD;

    BEGIN
	p := BufferAddress;
	FOR j := 1 TO count DO
	    SYSTEM.OutW (port, p^);
	    Lib.IncAddr (p, 2);
	END (*FOR*);
    END OutStringWord;

(************************************************************************)

END LowLevel.
