IMPLEMENTATION MODULE Graphics;

	(********************************************************)
	(*							*)
	(*		Screen graphics output.			*)
	(*							*)
	(*  Programmer:		P. Moylan			*)
	(*	VGA and VESA graphics support based on code	*)
	(*	developed by Luke Plaizier and Warren Reynolds.	*)
	(*  Last edited:	6 December 1993			*)
	(*  Status:		OK				*)
	(*	Hercules, CGA, EGA, and VGA modes working.	*)
	(*	SVGA seems to be working now.			*)
	(*							*)
	(*	The next obvious job is to improve character	*)
	(*	  string output.  One possible approach is	*)
	(*	  contained in procedure NewClippedString.	*)
	(*							*)
	(*	Not as fast as I would like - characters	*)
	(*	  especially slow, and line-plotting and Fill	*)
	(*	  could also do with some speed-up.  It's	*)
	(*	  likely that I am already close to the limit	*)
	(*	  of what can be achieved in terms of improving	*)
	(*	  the algorithms, and that the only remaining	*)
	(*	  source of speed-up would be a translation	*)
	(*	  into assembly language.  I'm reluctant to do	*)
	(*	  that until I believe that a stable version	*)
	(*	  has been reached.				*)
	(*							*)
	(********************************************************)

(************************************************************************)
(*									*)
(*  A particular problem in supporting screen operations on PC or AT	*)
(*  compatibles is that there are enormous differences between models	*)
(*  in what sort of graphics interface is provided.  Some of the	*)
(*  lowest-level part of this problem is handled by module Screen.	*)
(*									*)
(*  Remark: There is a considerable amount of code duplication in this	*)
(*  module.  This is deliberate.  Things need to be done slightly	*)
(*  differently in different modes, and we can gain some time by using	*)
(*  separate procedures for the different modes rather than putting	*)
(*  the decision logic inside a single general-purpose procedure.	*)
(*  Since graphics is expensive in computer time, this space/time	*)
(*  tradeoff can be worthwhile.  We haven't gone all the way in this	*)
(*  direction - that would make the module enormous - but the		*)
(*  separation has been done for those cases where the decision logic	*)
(*  was becoming unreasonably long.					*)
(*									*)
(*  For special applications, especially where it is known a priori	*)
(*  that certain modes will never be used, there is still some scope	*)
(*  for tuning this code.						*)
(*									*)
(************************************************************************)

FROM SYSTEM IMPORT
    (* type *)	BYTE;

FROM LowLevel IMPORT
    (* proc *)	MakePointer, Virtual, AddOffset, SubtractOffset, OutByte,
		IANDB, INOTB, RSB, LSB, Mul, BlockFill, BlockFillWord,
		OFFSET, HighWord, LowWord, INCV, DECV;

FROM Screen IMPORT
    (* type *)	VideoAdaptorType,
    (* proc *)	VideoKind, SetVideoMode, GetModeData,
		GetExtendedModeInformation, SelectReadBank, SelectWriteBank;

(************************************************************************)

VAR

    (* The type of video adaptor which is installed.	*)

    AdaptorKind: VideoAdaptorType;

    (* Our opinion of the "best" modes this adaptor supports.	*)

    DefaultTextMode, DefaultGraphicsMode: CARDINAL;

    (* Segment for the memory-mapped screen, and the starting port	*)
    (* number for the video I/O ports.  Defined as variables because	*)
    (* they depend on the adaptor type and current video mode.		*)
    (* NOTE: IObase not used so far in this module.  It's here solely	*)
    (* because it is obtained as a side-effect of getting ScreenSeg.	*)

    ScreenSeg, IObase: CARDINAL;

    (* Pointer to the currently active font table, for drawing text.	*)

    FontAddress: POINTER TO BYTE;

    (* Information about the current graphics mode.	*)

    ModeData:	RECORD
		    mode: CARDINAL;
		    MaxX, MaxY: CARDINAL;
		    MaxColour: ColourType;
		    BitsPerPixel: SHORTCARD;
		    BytesPerRow, FramesPerScreen: CARDINAL;
		    MultiBank: BOOLEAN;
		END (*RECORD*);

TYPE
    FillProcType = PROCEDURE (CARDINAL,CARDINAL,CARDINAL,CARDINAL,ColourType);
    LineProcType = PROCEDURE (CARDINAL, CARDINAL, INTEGER,
			CARDINAL, CARDINAL, CARDINAL, CARDINAL,
			BOOLEAN, ColourType);

VAR
    (* Procedure to perform the "Fill" operation. *)

    FillProc: FillProcType;

    (* Procedure to plot a straight line. *)

    VisibleLine: LineProcType;

(************************************************************************)
(*			  SPECIAL COLOUR MODES				*)
(************************************************************************)

TYPE
    BlackOrWhite = ColourType [0..1];
    CGAColour = ColourType [0..3];
    EGAColour = ColourType [0..15];

(************************************************************************)
(*		 	MASKS FOR BIT OPERATIONS			*)
(************************************************************************)

    (* A Mask value is used for stripping out a pixel from a byte, and	*)
    (* a Fill value is for filling a byte with one colour.		*)
    (* A Mask array is indexed by the pixel position (left to right)	*)
    (* within the byte, and a Fill array is indexed by colour.		*)

TYPE
    B2M = ARRAY [0..1] OF BYTE;
    B4M = ARRAY [0..3] OF BYTE;
    B8M = ARRAY [0..7] OF BYTE;

    B4C = ARRAY CGAColour OF BYTE;
    B2C = ARRAY BlackOrWhite OF BYTE;
    B16 = ARRAY EGAColour OF BYTE;

CONST
    (* Monochrome	*)

    Mask2 = B8M (80H, 40H, 20H, 10H, 8, 4, 2, 1);
    Fill2 = B2C (0, 0FFH);

    (* Four-colour palette: CGA colour	*)

    Mask4 = B4M (0C0H,30H,0CH,03H);
    Fill4 = B4C (0, 55H, 0AAH, 0FFH);

    (* Two pixels per byte: no such mode exists as far as I know, but	*)
    (* I'm playing it safe.						*)

    Mask16 = B2M (0F0H, 0FH);
    Fill16 = B16 (0, 11H, 22H, 33H, 44H, 55H, 66H, 77H, 88H, 99H,
			0AAH, 0BBH, 0CCH, 0DDH, 0EEH, 0FFH);

(************************************************************************)
(*		THE BASIC GRAPHICS OPERATION - PLOTTING A DOT		*)
(************************************************************************)

PROCEDURE PlotDot64K (x, y: CARDINAL;  colour: ColourType);

    (* Writes a dot at screen position (x, y).  This procedure is for	*)
    (* the direct colour modes, i.e. those which use a full word per	*)
    (* pixel.								*)

    VAR screenloc: POINTER TO CARDINAL;
	M1: CARDINAL;
	PixelLocation: LONGCARD;

    BEGIN
	WITH ModeData DO
	    M1 := BytesPerRow;
	    IF colour > MaxColour THEN colour := MaxColour END(*IF*);
	    y := MaxY - y;
	END (*WITH*);

	PixelLocation := Mul(M1,y) + Mul(2,x);
	SelectWriteBank (HighWord(PixelLocation));
	screenloc := MakePointer(ScreenSeg, LowWord (PixelLocation));
	screenloc^ := colour;

    END PlotDot64K;

(************************************************************************)

PROCEDURE PlotDot (x, y: CARDINAL;  colour: ColourType);

    (* Writes a dot at screen position (x, y).  The current version of	*)
    (* this procedure handles all modes which use one byte or less per	*)
    (* pixel.								*)

    VAR mask, fill: BYTE;
	screenloc: POINTER TO BYTE;
	DotsPerByte, M1, D1, bank: CARDINAL;
	PixelLocation: LONGCARD;

    BEGIN
	WITH ModeData DO
	    IF BitsPerPixel > 8 THEN
		PlotDot64K (x, y, colour);
		RETURN;
	    END (*IF*);
	    M1 := BytesPerRow;  D1 := FramesPerScreen;
	    DotsPerByte := 8 DIV VAL(CARDINAL,BitsPerPixel);
	    IF colour > MaxColour THEN colour := MaxColour END(*IF*);

	    IF MaxColour = 15 THEN
		mask := Mask2 [x MOD DotsPerByte];
		(* fill is unused in 16-colour modes *)
	    ELSIF DotsPerByte = 4 THEN
		mask := Mask4 [x MOD DotsPerByte];
		IF MaxColour = 1 THEN
		    fill := Fill2 [colour];
		ELSE
		    fill := Fill4 [colour];
		END (*IF*);
	    ELSIF DotsPerByte = 2 THEN
		mask := Mask16 [x MOD DotsPerByte];
		fill := Fill16 [colour];
	    ELSIF DotsPerByte > 1 THEN
		mask := Mask2 [x MOD DotsPerByte];
		fill := Fill2 [colour];
	    END (*IF*);

	    y := MaxY - y;
	END (*WITH*);

	(* Turn the (x,y) coordinates into a video memory address.	*)

	IF ModeData.MultiBank THEN

	    (* This part uses the 4F05 VESA function to switch the bank	*)
	    (* and then makes the screenloc pointer point to the	*)
	    (* location we need.					*)

	    PixelLocation := Mul(M1,y) + LONGCARD(x DIV DotsPerByte);

	    bank := HighWord(PixelLocation);
	    SelectWriteBank (bank);
	    IF DotsPerByte > 1 THEN
		SelectReadBank (bank);
	    END (*IF*);
	    screenloc := MakePointer(ScreenSeg, LowWord (PixelLocation));
	ELSE
	    screenloc:= MakePointer (ScreenSeg, M1*(y DIV D1)
				+ 8192*(y MOD D1) + x DIV DotsPerByte);
	END (*IF*);

	(* Clear out the old pixel value and set a new value.  For the	*)
	(* 16-colour modes, we do this by loading the colour code into	*)
	(* the set/reset register, the mask into the bit mask register,	*)
	(* and reading then writing the video memory location.  (The	*)
	(* actual data read and written are irrelevant, since the	*)
	(* actual data are taken from the graphics controller's		*)
	(* internal 32-bit latch.)  For the 256-colour modes, an 8-bit	*)
	(* colour code is written directly and no masking is needed.	*)
	(* For other modes, we have to do the masking ourselves.	*)

	IF ModeData.MaxColour = 15 THEN
	    OutByte (3CEH, 0);  OutByte (3CFH, BYTE(colour));
	    OutByte (3CEH, 8);  OutByte (3CFH, mask);
	    screenloc^ := screenloc^;
	ELSIF ModeData.MaxColour = 255 THEN
	    screenloc^ := VAL(BYTE,colour);
	ELSE
	    screenloc^ := IANDB (screenloc^, INOTB(mask)) + IANDB (mask, fill);
	END (*IF*);

    END PlotDot;

(************************************************************************)
(*			FILLING A RECTANGULAR REGION			*)
(************************************************************************)

PROCEDURE Fill256 (x0, y0, x1, y1: CARDINAL;  colour: ColourType);

    (* Fills the rectangle whose bottom left corner is (x0,y0) and	*)
    (* whose top right corner is (x1,y1) with the indicated colour.	*)
    (* This is the "fill" procedure for the modes supporting 256 or	*)
    (* more colours.							*)

    (* NOTE: This is a fairly rough job.  Needs to be looked at again	*)
    (* in terms of clarity and efficiency.				*)

    CONST MaxOffset = 0FFFFH;

    VAR screenloc: POINTER TO BYTE;
	bank, count, y, M1, BytesPerPixel, offset, bytesleft: CARDINAL;
	PixelLocation: LONGCARD;

    (********************************************************************)

    PROCEDURE StepForward (K: CARDINAL);

    (* Steps the screen location to K bytes beyond the current point.	*)

    BEGIN
	IF INCV (offset, K) THEN
	    INC (bank);
	    SelectWriteBank (bank);
	END (*IF*);
	screenloc := MakePointer(ScreenSeg, offset);
    END StepForward;

    (********************************************************************)

    BEGIN
	WITH ModeData DO
	    y0 := MaxY - y0;
	    y1 := MaxY - y1;
	    M1 := BytesPerRow;
	    BytesPerPixel := VAL(CARDINAL,BitsPerPixel) DIV 8;
	END (*WITH*);

	count := x1 - x0 + 1;
	PixelLocation := Mul(M1,y1) + Mul(BytesPerPixel,x0);
	bank := HighWord(PixelLocation);
	offset := LowWord (PixelLocation);
	SelectWriteBank (bank);

	FOR y := y1 TO y0 DO
	    screenloc := MakePointer(ScreenSeg, offset);

	    (* Slightly faked calculation below, as a temporary(?)	*)
	    (* expedient.						*)

	    IF offset = 0 THEN bytesleft := MaxOffset
	    ELSE bytesleft := MaxOffset - offset + 1;
	    END (*IF*);

	    IF BytesPerPixel*count <= bytesleft THEN
		IF BytesPerPixel = 1 THEN
		    BlockFill (screenloc, count, VAL(BYTE,colour));
		ELSE
		    BlockFillWord (screenloc, count, colour);
		END (*IF*);
		StepForward (M1);
	    ELSE
		IF BytesPerPixel = 1 THEN
		    BlockFill (screenloc, bytesleft, VAL(BYTE,colour));
		ELSE
		    BlockFillWord (screenloc, bytesleft DIV 2, colour);
		END (*IF*);
		StepForward (bytesleft);
		IF BytesPerPixel = 1 THEN
		    BlockFill (screenloc, count-bytesleft, VAL(BYTE,colour));
		ELSE
		    BlockFillWord (screenloc, count - bytesleft DIV 2, colour);
		END (*IF*);
		StepForward (M1-bytesleft);
	    END (*IF*);
	END (*FOR*);

    END Fill256;

(************************************************************************)

PROCEDURE MultibankEGAFill (ytop, ybottom: CARDINAL;  Lmask, Rmask: BYTE;
					firstcol, middlecount: CARDINAL);

    (* This procedure does part of the work for EGAFill - see below -	*)
    (* in cases where memory bank switching is required.		*)

    CONST DotsPerByte = 8;  MaxOffset = 0FFFFH;

    VAR screenloc: POINTER TO BYTE;
	BaseLocation: LONGCARD;
	y, M1, bank, offset, bytesleft: CARDINAL;

    (********************************************************************)

    PROCEDURE StepForward (K: CARDINAL);

    (* Steps the screen location to K bytes beyond the current point.	*)

    BEGIN
	IF INCV (offset, K) THEN
	    INC (bank);
	    SelectWriteBank (bank);
	END (*IF*);
	screenloc := MakePointer(ScreenSeg, offset);
    END StepForward;

    (********************************************************************)

    BEGIN
	(* Calculate the starting address in the video memory.  In what	*)
	(* follows, the actual data read and written are irrelevant;	*)
	(* the actual data are taken from the set/reset register (which	*)
	(* has already been loaded by the caller), and the pixels	*)
	(* affected are controlled by the bit mask register.		*)

	M1 := ModeData.BytesPerRow;
	BaseLocation := Mul(M1,ytop) + LONGCARD(firstcol);
	bank := HighWord (BaseLocation);
	offset := LowWord (BaseLocation);
	SelectWriteBank (bank);
	screenloc := MakePointer(ScreenSeg, offset);

	(* Redefine M1 to be the byte step between the end of a row	*)
	(* and the start of the next row.				*)

	DEC (M1, middlecount);
	IF Lmask <> BYTE(0FFH) THEN DEC(M1) END(*IF*);

	(* We go around the following loop once for each row. *)

	FOR y := ytop TO ybottom DO

	    (* Draw the dots in the first partial column, if any.	*)

	    IF Lmask <> BYTE(0FFH) THEN
		SelectReadBank (bank);
		OutByte (3CFH, Lmask);
		screenloc^ := screenloc^;
		StepForward(1);
	    END (*IF*);

	    (* Fill up the middle columns.	*)

	    IF middlecount > 0 THEN
		OutByte (3CFH, 0FFH);

		(* Slightly faked calculation below, as a temporary(?)	*)
		(* expedient.  It works because the case offset=0 will	*)
		(* never require a bank switch in the middle of a line.	*)

		IF offset = 0 THEN bytesleft := MaxOffset
		ELSE bytesleft := MaxOffset - offset + 1;
		END (*IF*);

		IF middlecount <= bytesleft THEN
		    BlockFill (screenloc, middlecount, 0);
		    StepForward (middlecount);
		ELSE
		    BlockFill (screenloc, bytesleft, 0);
		    StepForward (bytesleft);
		    BlockFill (screenloc, middlecount-bytesleft, 0);
		    StepForward (middlecount-bytesleft);
		END (*IF*);
	    END (*IF*);

	    (* Draw the dots in the last partial column, if any.	*)

	    IF Rmask <> BYTE(0FFH) THEN
		SelectReadBank (bank);
		OutByte (3CFH, Rmask);
		screenloc^ := screenloc^;
	    END (*IF*);

	    (* Move to the start of the next row. *)

	    StepForward (M1);

	END (*FOR*);

    END MultibankEGAFill;

(************************************************************************)

PROCEDURE EGAFill (x0, y0, x1, y1: CARDINAL;  colour: ColourType);

    (* Fills the rectangle whose bottom left corner is (x0,y0) and	*)
    (* whose top right corner is (x1,y1) with the indicated colour.	*)
    (* This procedure is used only for the 16-colour modes, for which	*)
    (* the video memory is arranged as 4 planes.  We treat these cases	*)
    (* separately since the method required is somewhat different than	*)
    (* for the other modes.						*)

    CONST DotsPerByte = 8;

    VAR Lmask, Rmask: BYTE;
	screenloc, baseloc: POINTER TO BYTE;
	firstcol, middlecount, y, M1: CARDINAL;

    BEGIN
	WITH ModeData DO
	    y0 := MaxY - y0;
	    y1 := MaxY - y1;
	    M1 := BytesPerRow;
	END (*WITH*);

	(* Work out the horizontal column range.  For our present	*)
	(* purposes, a "column" is a group of pixels which fit into one	*)
	(* byte.							*)

	firstcol := x0 DIV DotsPerByte;
	middlecount := x1 DIV DotsPerByte - firstcol;

	(* Work out the bit mask values for the left and right edges.	*)

	Lmask := RSB (0FFH, x0 MOD DotsPerByte);
	Rmask := LSB (0FFH, 7 - x1 MOD DotsPerByte);
	IF middlecount = 0 THEN
	    Lmask := IANDB (Lmask, Rmask);
	    Rmask := 0FFH;
	END (*IF*);

	IF Lmask = BYTE(0FFH) THEN
	    INC (middlecount);
	END (*IF*);

	IF Rmask <> BYTE(0FFH) THEN
	    DEC (middlecount);
	END (*IF*);

	(* Load the colour code into the set/reset register, then	*)
	(* select the bit mask register for all future operations.	*)

	OutByte (3CEH, 0);  OutByte (3CFH, VAL(BYTE,colour));
	OutByte (3CEH, 8);

	(* For modes where bank switching is required, the rest of this	*)
	(* job is done by a separate procedure.				*)

	IF ModeData.MultiBank THEN
	    MultibankEGAFill (y1, y0, Lmask, Rmask, firstcol, middlecount);
	    RETURN;
	END (*IF*);

	(* Turn the (x0,y1) coordinates into an address in the video	*)
	(* memory.  In what follows, the actual data read and written	*)
	(* are irrelevant; the actual data are taken from the set/reset	*)
	(* register, and the pixels affected are controlled by the	*)
	(* bit mask register.						*)

	baseloc := MakePointer (ScreenSeg, M1*y1 + firstcol);
	screenloc := baseloc;

	(* Draw the strip in the first partial column, if any.	*)

	IF Lmask <> BYTE(0FFH) THEN
	    OutByte (3CFH, Lmask);
	    FOR y := y1 TO y0 DO
		screenloc^ := screenloc^;
		screenloc := AddOffset (screenloc, M1);
	    END (*FOR*);
	    baseloc := AddOffset (baseloc, 1);
	    screenloc := baseloc;
	END (*IF*);

	(* Fill up the middle columns.	*)

	IF middlecount > 0 THEN
	    OutByte (3CFH, 0FFH);
	    FOR y := y1 TO y0 DO
		BlockFill (screenloc, middlecount, 0);
		screenloc := AddOffset (screenloc, M1);
	    END (*IF*);
	    baseloc := AddOffset (baseloc, middlecount);
	    screenloc := baseloc;
	END (*IF*);

	(* Draw the strip in the last partial column, if any.	*)

	IF Rmask <> BYTE(0FFH) THEN
	    OutByte (3CFH, Rmask);
	    FOR y := y1 TO y0 DO
		screenloc^ := screenloc^;
		screenloc := AddOffset (screenloc, M1);
	    END (*FOR*);
	END (*IF*);

    END EGAFill;

(************************************************************************)

PROCEDURE Fill0 (x0, y0, x1, y1: CARDINAL;  colour: ColourType);

    (* Fills the rectangle whose bottom left corner is (x0,y0) and	*)
    (* whose top right corner is (x1,y1) with the indicated colour.	*)
    (* This is the version of the Fill operation used for modes with	*)
    (* MaxColour<15 (and no bank switching).				*)

    VAR Lmask, Rmask, fillvalue: BYTE;
	Lfill, Rfill: SHORTCARD;
	screenloc: POINTER TO BYTE;
	firstcol, middlecount, M1, D1, y, DotsPerByte: CARDINAL;
	PixelSize: [1..8];

    BEGIN
	WITH ModeData DO
	    M1 := BytesPerRow;  D1 := FramesPerScreen;
	    DotsPerByte := 8 DIV VAL(CARDINAL,BitsPerPixel);
	    y0 := MaxY - y0;
	    y1 := MaxY - y1;
	    IF MaxColour = 1 THEN
		fillvalue := Fill2[colour];
	    ELSE
		fillvalue := Fill4[colour];
	    END (*IF*);
	END (*WITH*);

	(* Work out the horizontal column range.  For our present	*)
	(* purposes, a "column" is a group of pixels which fit into one	*)
	(* byte.							*)

	firstcol := x0 DIV DotsPerByte;
	middlecount := x1 DIV DotsPerByte - firstcol;

	(* Work out the mask values for the left and right edges.	*)
	(* Note: in the current version, each mask is the complement	*)
	(* of the corresponding mask in EGAFill.  I have considered	*)
	(* changing my conventions, but it looks as if the hardware	*)
	(* differences justify keeping this inconsistency.		*)

	PixelSize := 8 DIV DotsPerByte;
	Lmask := LSB (255, PixelSize*(DotsPerByte - x0 MOD DotsPerByte));
	Rmask := RSB (255, PixelSize*(1 + x1 MOD DotsPerByte));
	IF middlecount = 0 THEN
	    INC (Lmask, Rmask);  Rmask := 0;
	END (*IF*);

	IF Lmask = BYTE(0) THEN
	    INC (middlecount);
	ELSE
	    Lfill := IANDB (INOTB(Lmask), fillvalue);
	END (*IF*);

	IF Rmask <> BYTE(0) THEN
	    DEC (middlecount);
	    Rfill := IANDB (INOTB(Rmask), fillvalue);
	END (*IF*);

	FOR y := y1 TO y0 DO

	    (* Turn the (x0,y) coordinates into an address in	*)
	    (* the video memory.				*)

	    screenloc:= MakePointer (ScreenSeg,
				M1*(y DIV D1) + 8192*(y MOD D1) + firstcol);

	    (* Fix up the bits in the first partial column, if any.	*)

	    IF Lmask <> BYTE(0) THEN
		screenloc^ := IANDB (screenloc^, Lmask) + Lfill;
		screenloc := AddOffset (screenloc, 1);
	    END (*IF*);

	    (* Fill up the middle columns.	*)

	    IF middlecount > 0 THEN
		BlockFill (screenloc, middlecount, fillvalue);
		screenloc := AddOffset (screenloc, middlecount);
	    END (*IF*);

	    (* Fix up the bits in the last partial column, if any.	*)

	    IF Rmask <> BYTE(0) THEN
		screenloc^ := IANDB (screenloc^, Rmask) + Rfill;
	    END (*IF*);

	END (*FOR*);

    END Fill0;

(************************************************************************)

PROCEDURE Fill (x0, y0, x1, y1: CARDINAL;  colour: ColourType);

    (* Fills the rectangle whose bottom left corner is (x0,y0) and	*)
    (* whose top right corner is (x1,y1) with the indicated colour.	*)

    BEGIN
	FillProc (x0, y0, x1, y1, colour);
    END Fill;

(************************************************************************)
(*                                                                      *)
(*               SETTING THE 256 COLOUR PALLETE                         *)
(*                                                                      *)
(*	Palette_Index = 1..256 index to palette register. This can be	*)
(*		set to any one of 256k colours using the first six   	*)
(*		bits of each of Red, Green and Blue. The procedure will	*)
(*		accept 8 bit colours, but the top two bits will be	*)
(*		masked out.						*)
(*									*)
(*	NOTE: This routine is for some reason not operational on ALL	*)
(*		video cards. So far tested on Trident 8900B and 8900C	*)
(*		but not functional on Trident 9000C. Yet to conduct	*)
(*		further tests, but the problem seems related to the	*)
(*		VESA driver in use, although the Palette set is not a	*)
(*		VESA function. There are BIOS calls that can be 	*)
(*		substituted for direct PEL register writing, but these  *)
(*		are slower and at present unclearly documented for a	*)
(*		closer analysis.					*)
(************************************************************************)

PROCEDURE SetPaletteColour (Palette_Index, Red, Green, Blue: SHORTCARD);

VAR old_clock_value: SHORTCARD;

BEGIN

(* First disable the screen before the palette change. *)
(*
        OutByte(03C4H,1);
        old_clock_value := InByte(03C5H);
        OutByte(03C5H,IANDB(old_clock_value,254));
*)

(* Now access the palette index, and then set it to the desired colour. *)
        OutByte(03C8H, Palette_Index);
        OutByte(03C9H, IANDB(Red,63));
        OutByte(03C9H, IANDB(Green,63));
        OutByte(03C9H, IANDB(Blue,63));

(* Re-enable the screen. *)
        OutByte(03C5H,old_clock_value);

END SetPaletteColour;

(************************************************************************)
(*									*)
(*			PLOTTING STRAIGHT LINES				*)
(*									*)
(************************************************************************)
(*									*)
(*  In the following group of procedures, a line is specified by giving	*)
(*  one point on it, a Boolean goingdown, and the slope deltay/deltax,	*)
(*  where deltax and deltay are nonnegative integers.  We always draw	*)
(*  lines from left to right, and the vertical direction is defined by	*)
(*  goingdown.  The "current point" is represented by a triple		*)
(*  (x,y,ScaledError).  The integer ScaledError is a measure of how far	*)
(*  the discretized point is from the true straight line; it is		*)
(*  implicitly calculated as						*)
(*		(y-y0)*deltax - (x-x0)*deltay				*)
(*  where (x0,y0) is a point which lies precisely on the line.  When	*)
(*  drawing a line which is partly hidden, we have to pass ScaledError	*)
(*  from one procedure to another, and for this to work we must use	*)
(*  the same deltax and deltay for each segment of the line.  This is	*)
(*  the main reason for passing deltax, deltay, and goingdown as	*)
(*  procedure parameters rather than calculating them internally from	*)
(*  the endpoints.							*)
(*									*)
(*  For efficiency, these procedures do not call PlotDot; instead, they	*)
(*  do the equivalent operations internally.  Although this produces	*)
(*  some code redundancy, the consequent gain in speed is worthwhile.	*)
(*									*)
(************************************************************************)

PROCEDURE VisibleLine64K (xcurrent, ycurrent: CARDINAL;  ScaledError: INTEGER;
			deltax, deltay, xlimit, ylimit: CARDINAL;
			goingdown: BOOLEAN;  colour: ColourType);

    (* Plots a straight line of slope deltay/deltax starting from	*)
    (* (xcurrent,ycurrent,ScaledError).  This procedure is used only	*)
    (* for the direct colour modes using two bytes per pixel.  For	*)
    (* details about the parameters, etc., see procedure VisibleLine0.	*)

    VAR xthreshold, ythreshold, OldError: INTEGER;
	screenloc: POINTER TO CARDINAL;
	M1, temp, bank, offset: CARDINAL;
	PixelLocation: LONGCARD;

    BEGIN

	WITH ModeData DO
	    M1 := BytesPerRow;
	    temp := MaxY - ycurrent;
	END (*WITH*);

	(* Plot the initial point. *)

	PixelLocation := Mul(M1,temp) + Mul(2,xcurrent);
	bank := HighWord(PixelLocation);
	SelectWriteBank (bank);
	offset := LowWord (PixelLocation);
	screenloc := MakePointer(ScreenSeg, offset);
	screenloc^ := colour;

	IF (deltax = 0) AND (deltay = 0) THEN RETURN END(*IF*);

	xthreshold := INTEGER(deltay DIV 2) - INTEGER(deltax);
	ythreshold := INTEGER(deltay) - INTEGER(deltax DIV 2);

	(* Now, here is the main part of the line algorithm.  Each time	*)
	(* around the loop we decide in which direction to move for the	*)
	(* next point. and then plot that point.			*)

	LOOP
	    OldError := ScaledError;
	    IF OldError > xthreshold THEN
		IF xcurrent = xlimit THEN EXIT(*LOOP*) END(*IF*);
		DEC (ScaledError, deltay);
		INC (xcurrent);
		IF INCV (offset, 2) THEN
		    INC (bank);
		    SelectWriteBank (bank);
		END (*IF*);
	    END (*IF*);
	    IF OldError < ythreshold THEN
		IF ycurrent = ylimit THEN EXIT(*LOOP*) END(*IF*);
		INC (ScaledError, deltax);
		IF goingdown THEN
		    DEC (ycurrent);
		    IF INCV (offset, M1) THEN
			INC (bank);
			SelectWriteBank (bank);
		    END (*IF*);
		ELSE			(* NOT goingdown*)
		    INC (ycurrent);
		    IF DECV (offset, M1) THEN
			DEC (bank);
			SelectWriteBank (bank);
		    END (*IF*);
		END (*IF*);
	    END (*IF*);

	    (* Plot the new point *)

	    screenloc := MakePointer (ScreenSeg, offset);
	    screenloc^ := colour;

	END (*LOOP*);

    END VisibleLine64K;

(************************************************************************)

PROCEDURE VisibleLine0 (xcurrent, ycurrent: CARDINAL;  ScaledError: INTEGER;
			deltax, deltay, xlimit, ylimit: CARDINAL;
			goingdown: BOOLEAN;  colour: ColourType);

    (* Plots a straight line of slope deltay/deltax starting from	*)
    (* (xcurrent,ycurrent,ScaledError).  The method used is a variant	*)
    (* of Bresenham's algorithm.  Stops plotting when xcurrent is about	*)
    (* to step beyond xlimit or when ycurrent is about to step beyond	*)
    (* ylimit.  It is assumed that the input data have been		*)
    (* pre-processed to ensure that the initial point does not violate	*)
    (* the bounds, and that there is no risk of plotting a point	*)
    (* outside the range of the screen.					*)

    (* This version of the procedure works for all modes which use	*)
    (* one byte or less per pixel.					*)

    CONST MaxOffset = MAX(CARDINAL);

    VAR xthreshold, ythreshold, OldError: INTEGER;
	mask0, mask, fill: BYTE;
	screenloc: POINTER TO BYTE;
	M1, D1, temp, frame, framewrap, bank, offset, DotsPerByte: CARDINAL;
	PixelLocation: LONGCARD;

    BEGIN

	WITH ModeData DO
	    M1 := BytesPerRow;  D1 := FramesPerScreen;
	    IF D1 <> 1 THEN
		framewrap := 8192*(D1-1) - M1;
	    END (*IF*);
	    DotsPerByte := 8 DIV VAL(CARDINAL,BitsPerPixel);

	    IF MaxColour = 15 THEN
		OutByte (3CEH, 0);  OutByte (3CFH, VAL(BYTE,colour));
		OutByte (3CEH, 8);
	    ELSIF MaxColour = 3 THEN
		fill := Fill4[VAL(CGAColour,colour)];
	    ELSIF MaxColour = 1 THEN
		fill := Fill2[VAL(BlackOrWhite,colour)];
	    END (*IF*);

	    IF BitsPerPixel = 2 THEN
		mask0 := 0C0H;
		mask := Mask4[xcurrent MOD DotsPerByte];
	    ELSE
		mask0 := 80H;
		mask := Mask2[xcurrent MOD DotsPerByte];
	    END (*IF*);

	    temp := MaxY - ycurrent;

	END (*WITH*);

	(* Work out the address of the initial point. *)

	IF ModeData.MultiBank THEN

	    (* This part uses the 4F05 VESA function to switch the bank	*)
	    (* and then makes the screenloc pointer point to the	*)
	    (* location we need.					*)

	    frame := 0;
	    PixelLocation := Mul(M1,temp) + LONGCARD(xcurrent DIV DotsPerByte);

	    bank := HighWord(PixelLocation);
	    SelectWriteBank (bank);
	    IF DotsPerByte > 1 THEN
		SelectReadBank (bank);
	    END (*IF*);
	    offset := LowWord (PixelLocation);
	    screenloc := MakePointer(ScreenSeg, offset);
	ELSE
	    frame := temp MOD D1;
	    offset := M1*(temp DIV D1) + 8192*frame + xcurrent DIV DotsPerByte;
	    screenloc:= MakePointer (ScreenSeg, offset);
	END (*IF*);

	(* Plot the initial point. *)

	IF ModeData.MaxColour = 15 THEN
	    OutByte (3CFH, mask);
	    screenloc^ := screenloc^;
	ELSIF ModeData.MaxColour = 255 THEN
	    screenloc^ := VAL(SHORTCARD,colour);
	ELSE
	    screenloc^ := IANDB (screenloc^, INOTB(mask))
					+ IANDB (mask, fill);
	END (*IF*);

	IF (deltax = 0) AND (deltay = 0) THEN RETURN END(*IF*);

	xthreshold := INTEGER(deltay DIV 2) - INTEGER(deltax);
	ythreshold := INTEGER(deltay) - INTEGER(deltax DIV 2);

	(* Now, here is the main part of the line algorithm.  Each time	*)
	(* around the loop we decide in which direction to move for the	*)
	(* next point. and then plot that point.			*)

	LOOP
	    OldError := ScaledError;
	    IF OldError > xthreshold THEN
		IF xcurrent = xlimit THEN EXIT(*LOOP*) END(*IF*);
		DEC (ScaledError, deltay);
		INC (xcurrent);
		mask := RSB (mask, VAL(CARDINAL,ModeData.BitsPerPixel));
		IF mask = BYTE(0) THEN
		    mask := mask0;
		    IF INCV (offset, 1) THEN
			INC (bank);
			SelectWriteBank (bank);
			IF DotsPerByte > 1 THEN
			    SelectReadBank (bank);
			END (*IF*);
		    END (*IF*);
		END (*IF*);
	    END (*IF*);
	    IF OldError < ythreshold THEN
		IF ycurrent = ylimit THEN EXIT(*LOOP*) END(*IF*);
		INC (ScaledError, deltax);
		IF goingdown THEN
		    DEC (ycurrent);
		    IF D1 = 1 THEN
			IF INCV (offset, M1) THEN
			    INC (bank);
			    SelectWriteBank (bank);
			    IF DotsPerByte > 1 THEN
				SelectReadBank (bank);
			    END (*IF*);
			END (*IF*);
		    ELSIF frame = D1-1 THEN
			frame := 0;
			DEC (offset, framewrap);
		    ELSE
			INC (frame);
			INC (offset, 8192);
		    END (*IF*);
		ELSE			(* NOT goingdown*)
		    INC (ycurrent);
		    IF D1 = 1 THEN
			IF DECV (offset, M1) THEN
			    DEC (bank);
			    SelectWriteBank (bank);
			    IF DotsPerByte > 1 THEN
				SelectReadBank (bank);
			    END (*IF*);
			END (*IF*);
		    ELSIF frame = 0 THEN
			frame := D1-1;
			INC (offset, framewrap);
		    ELSE
			DEC (frame);
			DEC (offset, 8192);
		    END (*IF*);
		END (*IF*);
	    END (*IF*);

	    (* Plot the new point *)

	    screenloc := MakePointer (ScreenSeg, offset);
	    IF ModeData.MaxColour = 255 THEN
		screenloc^ := VAL(SHORTCARD,colour);
	    ELSIF ModeData.MaxColour = 15 THEN
		OutByte (3CFH, mask);
		screenloc^ := screenloc^;
	    ELSE
		screenloc^ := IANDB (screenloc^, INOTB(mask))
						+ IANDB (mask, fill);
	    END (*IF*);

	END (*LOOP*);

    END VisibleLine0;

(************************************************************************)
(*
PROCEDURE OldVisibleLine (xcurrent, ycurrent: CARDINAL;  ScaledError: INTEGER;
			deltax, deltay, xlimit, ylimit: CARDINAL;
			goingdown: BOOLEAN;  colour: CARDINAL);

    (* Plots a straight line of slope deltay/deltax starting from	*)
    (* (xcurrent,ycurrent,ScaledError).  The method used is a variant	*)
    (* of Bresenham's algorithm.  Stops plotting when xcurrent is about	*)
    (* to step beyond xlimit or when ycurrent is about to step beyond	*)
    (* ylimit.  It is assumed that the input data have been		*)
    (* pre-processed to ensure that the initial point does not violate	*)
    (* the bounds, and that there is no risk of plotting a point	*)
    (* outside the range of the screen.					*)

    (* This procedure is no longer used, but is left here as internal	*)
    (* documentation: it uses the same logic as the more efficient	*)
    (* versions above, and is easier to read.				*)

    VAR xthreshold, ythreshold, OldError: INTEGER;

    BEGIN

	(* Plot the initial point. *)

	PlotDot (xcurrent, ycurrent, colour);
	IF (deltax = 0) AND (deltay = 0) THEN RETURN END(*IF*);

	xthreshold := INTEGER(deltay DIV 2) - INTEGER(deltax);
	ythreshold := INTEGER(deltay) - INTEGER(deltax DIV 2);

	(* Now, here is the main part of the line algorithm.  Each time	*)
	(* around the loop we decide in which direction to move for the	*)
	(* next point. and then plot that point.			*)

	LOOP
	    OldError := ScaledError;
	    IF OldError > xthreshold THEN
		IF xcurrent = xlimit THEN EXIT(*LOOP*) END(*IF*);
		DEC (ScaledError, deltay);
		INC (xcurrent);
	    END (*IF*);
	    IF OldError < ythreshold THEN
		IF ycurrent = ylimit THEN EXIT(*LOOP*) END(*IF*);
		INC (ScaledError, deltax);
		IF goingdown THEN
		    DEC (ycurrent);
		ELSE
		    INC (ycurrent);
		END (*IF*);
	    END (*IF*);
	    PlotDot (xcurrent, ycurrent, colour);
	END (*LOOP*);

    END OldVisibleLine;
*)
(************************************************************************)

PROCEDURE MoveToX (x0, y0, deltax, deltay, X: CARDINAL;  goingdown: BOOLEAN;
			VAR (*OUT*) ScaledError: INTEGER): CARDINAL;

    (* For the line starting at (x0,y0) of slope deltay/deltax, returns	*)
    (* the first y for which the discrete approximation to the line	*)
    (* hits x=X.  Also calculates the ScaledError at that point.	*)

    VAR result: LONGINT;
	longX, longdx, longdy: LONGINT;

    BEGIN
	(* Shift the origin. *)

	DEC (X, x0);

	(* The calculations below produce CARDINAL or INTEGER results,	*)
	(* but a greater range is needed for the temporary values after	*)
	(* each multiplication.  This is one of those unfortunate cases	*)
	(* where something which can be expressed clearly and concisely	*)
	(* in assembly language becomes obscure when written in a	*)
	(* high-level language.  The Modula-2 rules on assignment	*)
	(* compatibility don't help, either.				*)

	longX := LONGCARD(X);
	longdx := LONGCARD(deltax);  longdy := LONGCARD(deltay);
	
	(* We have to use a different method depending on whether the	*)
	(* line slope is greater or less than 1.			*)

	IF deltay <= deltax THEN
	    result := (2*longX*longdy + longdx - 1) DIV (2*longdx)
	ELSE
	    result := (2*longX - 1)*longdy DIV (2*longdx) + 1
	END (*IF*);

	ScaledError := INTEGER(result*longdx - longX*longdy);

	(* Reverse the origin shift. *)

	IF goingdown THEN RETURN y0-CARDINAL(result)
	ELSE RETURN y0+CARDINAL(result);
	END (*IF*);

    END MoveToX;

(************************************************************************)

PROCEDURE MoveToY (x0, y0, deltax, deltay, Y: CARDINAL;  goingdown: BOOLEAN;
			VAR (*OUT*) ScaledError: INTEGER): CARDINAL;

    (* For the line starting at (x0,y0) of slope deltay/deltax, returns	*)
    (* the first x for which the discrete approximation to the line	*)
    (* hits y=Y.  Also calculates the ScaledError at that point.	*)

    VAR result: LONGINT;
	longY, longdx, longdy: LONGINT;

    BEGIN
	(* Shift the origin. *)

	IF goingdown THEN Y := y0 - Y ELSE DEC (Y, y0) END(*IF*);

	longY := LONGINT(Y);
	longdx := LONGINT(deltax);  longdy := LONGINT(deltay);

	(* We have to use a different method depending on whether the	*)
	(* line slope is greater or less than 1.			*)

	IF deltay <= deltax THEN
	    result := (2*longY - 1)*longdx DIV (2*longdy) + 1
	ELSE
	    result := (2*longY*longdx + longdy - 1) DIV (2*longdy)
	END (*IF*);
	ScaledError := INTEGER(longY*longdx - result*longdy);

	(* Reverse the origin shift. *)

	RETURN VAL(CARDINAL,result)+x0;

    END MoveToY;

(************************************************************************)

PROCEDURE PlotLine (x0, y0, x1, y1: CARDINAL;  colour: ColourType);

    (* Plots a straight line from (x0,y0) to (x1,y1).  It is the	*)
    (* caller's responsibility to ensure that the coordinates are in	*)
    (* range for the current video mode.				*)

    VAR temp, deltay: CARDINAL;
	goingdown: BOOLEAN;

    BEGIN

	(* First, ensure that we are working in the +X direction.	*)

	IF x1 < x0 THEN
	    temp := x1;  x1 := x0;  x0 := temp;
	    temp := y1;  y1 := y0;  y0 := temp;
	END (*IF*);

	(* Check the Y direction. *)

	IF y1 >= y0 THEN
	    goingdown := FALSE;  deltay := y1 - y0;
	ELSE
	    goingdown := TRUE;  deltay := y0 - y1;
	END (*IF*);

	(* Draw the line.*)

	VisibleLine (x0, y0, 0, x1-x0, deltay, x1, y1, goingdown, colour);

    END PlotLine;

(************************************************************************)

PROCEDURE PlotRectangle (R: Rectangle;  colour: ColourType);

    (* Plots a rectangle, with clipping if necessary to keep the	*)
    (* points within the screen boundary.				*)

    VAR leftOK, rightOK, topOK, bottomOK: BOOLEAN;

    BEGIN
	WITH R DO
	    WITH ModeData DO
		IF (left > INTEGER(MaxX)) OR (right < 0)
			OR (bottom > INTEGER(MaxY)) OR (top < 0) THEN RETURN;
		END (*IF*);
		leftOK := TRUE;  rightOK := TRUE;
		topOK := TRUE;  bottomOK := TRUE;
		IF left < 0 THEN
		    left := 0;  leftOK := FALSE;
		END(*IF*);
		IF right > INTEGER(MaxX) THEN
		    right := INTEGER(MaxX);  rightOK := FALSE;
		END(*IF*);
		IF bottom < 0 THEN
		    bottom := 0;  bottomOK := FALSE;
		END(*IF*);
		IF top > INTEGER(MaxY) THEN
		    top := INTEGER(MaxY);  topOK := FALSE;
		END(*IF*);
	    END (*WITH*);
	    IF leftOK THEN
		PlotLine (left, bottom, left, top, colour);
	    END(*IF*);
	    IF rightOK THEN
		PlotLine (right, bottom, right, top, colour);
	    END(*IF*);
	    IF bottomOK THEN
		PlotLine (left, bottom, right, bottom, colour);
	    END(*IF*);
	    IF topOK THEN
		PlotLine (left, top, right, top, colour);
	    END(*IF*);
	END (*WITH*);
    END PlotRectangle;

(************************************************************************)

PROCEDURE ClippedLine (x0, y0, x1, y1: CARDINAL;  colour: ColourType;
			left, right, bottom, top: CARDINAL);

    (* Like PlotLine, but plots only that part of the line which lies	*)
    (* in the rectangle (left <= x <= right), (bottom <= y <= top).	*)
    (* The caller is expected to ensure, by appropriate definition of	*)
    (* the rectangle, that all plotted points are in range for the	*)
    (* current video mode.						*)

    VAR temp, deltax, deltay, xlimit, ylimit: CARDINAL;
	goingdown: BOOLEAN;
	ScaledError: INTEGER;

    BEGIN

	(* First, ensure that we are working in the +X direction; and	*)
	(* check the Y direction.					*)

	IF x1 < x0 THEN
	    temp := x1;  x1 := x0;  x0 := temp;
	    temp := y1;  y1 := y0;  y0 := temp;
	END (*IF*);
	goingdown := y1 < y0;

	(* Eliminate some (but not all) cases where we are going to	*)
	(* miss the rectangle entirely.  Also calculate the slope and	*)
	(* boundary parameters.						*)

	IF x1 < left THEN RETURN END(*IF*);
	ylimit := y1;
	IF goingdown THEN
	    IF (y0 < bottom) OR (y1 > top) THEN RETURN END(*IF*);
	    deltay := y0 - y1;
	    IF ylimit < bottom THEN ylimit := bottom END(*IF*);
	ELSE
	    IF (y0 > top) OR (y1 < bottom) THEN RETURN END(*IF*);
	    deltay := y1 - y0;
	    IF ylimit > top THEN ylimit := top END(*IF*);
	END (*IF*);

	deltax := x1 - x0;
	xlimit := x1;
	IF xlimit > right THEN xlimit := right END(*IF*);

	(* We've now extracted all we need to know about the target	*)
	(* point.  From here on, we use (x1,y1,ScaledError) to		*)
	(* represent the current point.					*)

	x1 := x0;  y1 := y0;  ScaledError := 0;

	(* Step up to the left boundary, if we're at the left of it.	*)

	IF x0 < left THEN
	    x1 := left;
	    y1 := MoveToX (x0, y0, deltax, deltay, left,
						goingdown, ScaledError);
	END (*IF*);

	(* We might not yet have hit the rectangle.	*)

	IF goingdown THEN
	    IF y1 < bottom THEN RETURN END(*IF*);
	    IF y1 > top THEN
		x1 := MoveToY (x0, y0, deltax, deltay, top,
					goingdown, ScaledError);
		y1 := top;
	    END (*IF*);
	ELSE
	    IF y1 > top THEN RETURN END(*IF*);
	    IF y1 < bottom THEN
		x1 := MoveToY (x0, y0, deltax, deltay, bottom,
					goingdown, ScaledError);
		y1 := bottom;
	    END (*IF*);
	END (*IF*);

	(* Check whether we missed the rectangle entirely.	*)

	IF x1 > right THEN RETURN END (*IF*);

	(* At last, we have something to plot.	*)

	VisibleLine (x1, y1, ScaledError, deltax, deltay,
					xlimit, ylimit, goingdown, colour);

    END ClippedLine;

(************************************************************************)
(*			PUTTING A MARK AT A POINT			*)
(************************************************************************)

PROCEDURE PlotMark (x, y: CARDINAL;
			colour: ColourType;  pointtype: SHORTCARD);

    (* Writes a symbol at screen position (x, y).	*)

    BEGIN
	CASE pointtype OF
	    1:	PlotLine (x-1,y-1,x+1,y+1, colour);	(*  X	*)
		PlotLine (x+1,y-1,x-1,y+1, colour);
	  |
	    2:	PlotLine (x-2,y-1,x+2,y-1, colour);	(* box	*)
		PlotLine (x+2,y-1,x+2,y+1, colour);
		PlotLine (x+2,y+1,x-2,y+1, colour);
		PlotLine (x-2,y+1,x-2,y-1, colour);
	  |
	    ELSE
		PlotDot (x, y, colour);			(* point *)
	END (*CASE*);
    END PlotMark;

(************************************************************************)
(*			    DRAWING CHARACTERS				*)
(************************************************************************)

PROCEDURE DrawChar (ch: CHAR;  x, y: CARDINAL;  colour: ColourType);

    (* Draws the single character ch.  The coordinates (x,y) are the	*)
    (* location of the bottom left of the character.			*)

    VAR fontptr: POINTER TO BYTE;  pattern: SHORTCARD;
	i, j: [0..7];  xstart: CARDINAL;

    BEGIN
	xstart := x + 7;  INC (y, 7);
	fontptr := AddOffset (FontAddress, 8*ORD(ch));
	FOR i := 0 TO 7 DO
	    pattern := fontptr^;  x := xstart;
	    FOR j := 0 TO 7 DO
		IF ODD(pattern) THEN
		    PlotDot (x, y, colour);
		END (*IF*);
		DEC (x);
		pattern := pattern DIV 2;
	    END (*FOR*);
	    DEC (y);
	    fontptr := AddOffset (fontptr, 1);
	END (*FOR*);
    END DrawChar;

(************************************************************************)

PROCEDURE ClippedChar (ch: CHAR;  x, y: CARDINAL;  colour: ColourType;
			left, right, bottom, top: CARDINAL);

    (* Like DrawChar, but excludes those parts of the character which	*)
    (* fall outside the given clip rectangle.				*)

    VAR fontptr: POINTER TO BYTE;  pattern, mask: BYTE;
	xstart: CARDINAL;

    BEGIN
	(* Eliminate the trivial cases.	*)

	IF (x > right) OR (x+7 < left) OR (y > top) OR (y+7 < bottom) THEN
	    RETURN;
	END (*IF*);

	xstart := x;  mask := 128;
	IF xstart+7 < right THEN right := xstart+7  END(*IF*);

	(* Check for the case of a character clipped at the left side.	*)

	IF xstart < left THEN
	    mask := RSB (mask, left - xstart);
	    xstart := left;
	END (*IF*);

	(* Establish the desired vertical range.	*)

	IF y > bottom THEN bottom := y END(*IF*);
	fontptr := AddOffset (FontAddress, 8*ORD(ch));
	INC (y, 7);

	(* Check for a character clipped at the top.	*)

	IF y > top THEN
	    fontptr := AddOffset (fontptr, y-top);  y := top;
	END (*IF*);

	(* The outer loop steps through bytes in the font table,	*)
	(* while stepping in the -Y direction.				*)

	LOOP
	    pattern := fontptr^;  x := xstart;

	    (* The inner loop steps in the +X direction while stepping	*)
	    (* through bits in the byte read from the font table.	*)

	    LOOP
		IF pattern = BYTE(0) THEN EXIT(*LOOP*) END(*IF*);
		IF IANDB (pattern, mask) <> 0 THEN
		    PlotDot (x, y, colour);
		END (*IF*);
		INC (x);
		IF x > right THEN EXIT(*LOOP*) END(*IF*);
		pattern := LSB (pattern, 1);
	    END (*LOOP*);

	    DEC (y);
	    IF y < bottom THEN EXIT(*LOOP*) END(*IF*);
	    fontptr := AddOffset (fontptr, 1);

	END (*LOOP*);

    END ClippedChar;

(************************************************************************)

PROCEDURE DrawCharUp (ch: CHAR;  x, y: CARDINAL;  colour: ColourType);

    (* Draws the single character ch sideways.  The coordinates (x,y)	*)
    (* are the location of the bottom left of the unrotated character,	*)
    (* or equivalently the bottom right of the character as plotted.	*)

    VAR fontptr: POINTER TO BYTE;  pattern: SHORTCARD;
	i, j: [0..7];

    BEGIN
	fontptr := AddOffset (FontAddress, 8*ORD(ch));
	FOR i := 0 TO 7 DO
	    pattern := fontptr^;
	    FOR j := 0 TO 7 DO
		IF ODD(pattern) THEN
		    PlotDot (x+i-7, y+7-j, colour);
		END (*IF*);
		pattern := pattern DIV 2;
	    END (*FOR*);
	    fontptr := AddOffset (fontptr, 1);
	END (*FOR*);
    END DrawCharUp;

(************************************************************************)

PROCEDURE ClippedCharUp (ch: CHAR;  x, y: CARDINAL;  colour: ColourType;
			left, right, bottom, top: CARDINAL);

    (* Like DrawCharUp, but excludes those parts of the character which	*)
    (* fall outside the given clip rectangle.				*)

    VAR fontptr: POINTER TO BYTE;  pattern, mask: BYTE;
	ystart: CARDINAL;

    BEGIN
	(* Eliminate the trivial cases.	*)

	IF (x > right+7) OR (x < left) OR (y > top) OR (y+7 < bottom) THEN
	    RETURN;
	END (*IF*);

	ystart := y;  mask := 128;
	IF ystart+7 < top THEN top := ystart+7 END(*IF*);

	(* Check for the case of a character clipped at the side.	*)

	IF ystart < bottom THEN
	    mask := RSB (mask, bottom - ystart);
	    ystart := bottom;
	END (*IF*);

	(* Establish the desired horizontal range.	*)

	IF x < right THEN right := x END(*IF*);
	fontptr := AddOffset (FontAddress, 8*ORD(ch));
	DEC (x, 7);

	(* Check for a character clipped at the top.	*)

	IF x < left THEN
	    fontptr := AddOffset (fontptr, left-x);  x := left;
	END (*IF*);

	(* The outer loop steps through bytes in the font table,	*)
	(* while stepping in the +X direction.				*)

	LOOP
	    pattern := fontptr^;  y := ystart;

	    (* The inner loop steps in the +Y direction while stepping	*)
	    (* through bits in the byte read from the font table.	*)

	    LOOP
		IF pattern = BYTE(0) THEN EXIT(*LOOP*) END(*IF*);
		IF IANDB (pattern, mask) <> 0 THEN
		    PlotDot (x, y, colour);
		END (*IF*);
		INC (y);
		IF y > top THEN EXIT(*LOOP*) END(*IF*);
		pattern := LSB (pattern, 1);
	    END (*LOOP*);

	    INC (x);
	    IF x > right THEN EXIT(*LOOP*) END(*IF*);
	    fontptr := AddOffset (fontptr, 1);

	END (*LOOP*);

    END ClippedCharUp;

(************************************************************************)

PROCEDURE PlotString (VAR (*IN*) text: ARRAY OF CHAR;
			x, y, length: CARDINAL;  colour: ColourType);

    (* Draws a string of "length" characters starting at location (x,y)	*)

    VAR j: CARDINAL;

    BEGIN
	FOR j := 0 TO length-1 DO
	    DrawChar (text[j], x, y, colour);
	    INC (x, 8);
	END (*FOR*);
    END PlotString;

(************************************************************************)

PROCEDURE PlotStringUp (VAR (*IN*) text: ARRAY OF CHAR;
			x, y, length: CARDINAL;  colour: ColourType);

    (* Like PlotString, but with text written in the +Y direction.	*)

    VAR j: CARDINAL;

    BEGIN
	FOR j := 0 TO length-1 DO
	    DrawCharUp (text[j], x, y, colour);
	    INC (y, 8);
	END (*FOR*);
    END PlotStringUp;

(************************************************************************)

PROCEDURE ClippedString (VAR (*IN*) text: ARRAY OF CHAR;
			x, y, length: CARDINAL;  colour: ColourType;
			left, right, bottom, top: CARDINAL);

    (* Like PlotString, but excludes any points which fall outside the	*)
    (* clip rectangle defined by (left,right,bottom,top).		*)

    (* Question: is this too elaborate?  Perhaps I can achieve greater	*)
    (* efficiency simply by doing a better job on ClippedChar.		*)

    VAR j, j0, j1, j2, endx: CARDINAL;
 	clipfirst, cliplast: BOOLEAN;

    BEGIN
(*
	(* New version: is this just as good?	*)

	FOR j := 0 TO length-1 DO
	    ClippedChar (text[j], x, y, colour, left, right, bottom, top);
	    INC (x, 8);
	END (*FOR*);
*)
	(* ORIGINAL VERSION STARTS HERE:				*)

	(* Subdivide the string as follows:				*)
	(*	0..j0-1		Nothing to plot				*)
	(*	j0..j1-1	Plot clipped				*)
	(*	j1..j2		Plot unclipped				*)
	(*	j2+1		Plot clipped iff cliplast = TRUE	*)
	(* Along the way we exclude some trivial cases.			*)

	IF (y >= top) OR (y+8 < bottom) THEN
	    RETURN;
	END (*IF*);

	j1 := 0;  j2 := length-1;
	endx := x + 8*length - 1;

	IF (x > right) OR (endx <= left) THEN
	    RETURN;
	END (*IF*);
	WHILE (x < left) DO
	    INC (j1);  INC (x, 8);
	END (*WHILE*);
	clipfirst := (x > left) AND (j1 > 0);

	WHILE (endx > right) DO
	    DEC (j2);  DEC (endx, 8);
	END (*WHILE*);
	cliplast := (endx < right) AND (j2 < length-1);

	(* This completes the horizontal processing.  Now check for	*)
	(* the string straddling the top or bottom boundary.		*)

	j0 := j1;
	IF (y > top-8) OR (y < bottom) THEN
	    j1 := j2+1;
	END (*IF*);

	IF clipfirst THEN
	    DEC (x, 8);  DEC(j0);
	END(*IF*);

	(* End of subdivision; now we can do the plotting.	*)

	IF j0 < j1 THEN
	    FOR j := j0 TO j1-1 DO
		ClippedChar (text[j], x, y, colour, left, right, bottom, top);
		INC (x, 8);
	    END (*FOR*);
	END (*IF*);

	IF j1 <= j2 THEN
	    FOR j := j1 TO j2 DO
		DrawChar (text[j], x, y, colour);
		INC (x, 8);
	    END (*FOR*);
	END (*IF*);

	IF cliplast THEN
	    ClippedChar (text[j2+1], x, y, colour, left, right, bottom, top);
	END (*IF*);

    END ClippedString;

(************************************************************************)

PROCEDURE NewClippedString (VAR (*IN*) text: ARRAY OF CHAR;
				x, y, length: CARDINAL;  colour: SHORTCARD;
				left, right, bottom, top: CARDINAL);

    (* Like PlotString, but excludes any points which fall outside the	*)
    (* clip rectangle defined by (left,right,bottom,top).		*)

    (* New approach: I'm going to try a 'raster' approach to the	*)
    (* plotting, by making eight passes over the entire string rather	*)
    (* than plotting a character at a time.				*)

    BEGIN
	(*NYI*)

	(* What follows is just rough notes so far.		*)
(*
	xstart := x + 7;  INC (y, 7);
	fontptr := FontAddress;
	screenptr := ScreenAddress (x, y);
*)
    END NewClippedString;

(************************************************************************)

PROCEDURE ClippedUpString (VAR (*IN*) text: ARRAY OF CHAR;
			x, y, length: CARDINAL;  colour: ColourType;
			left, right, bottom, top: CARDINAL);

    (* Like ClippedString, but with text written in the +Y direction.	*)

    VAR j: CARDINAL;

    BEGIN
	FOR j := 0 TO length-1 DO
	    ClippedCharUp (text[j], x, y, colour, left, right, bottom, top);
	    INC (y, 8);
	END (*FOR*);
    END ClippedUpString;

(************************************************************************)
(*		      OPERATIONS ON THE VIDEO MODE			*)
(************************************************************************)

PROCEDURE SetMode (newmode: CARDINAL;  ClearScreen: BOOLEAN);

    (* Sets the video mode. *)

    VAR BitsPerPixel: SHORTCARD;  ColourLimit: CARDINAL;

    BEGIN
	IF NOT SetVideoMode (newmode, ClearScreen) THEN
	    RETURN;
	END (*IF*);
	GetModeData (ScreenSeg, IObase);
	WITH ModeData DO
	    mode := newmode;  MultiBank := FALSE;
	    CASE newmode OF
		HercGraphics:
			BitsPerPixel := 1;
			BytesPerRow := 90;  FramesPerScreen := 4;
			MaxX := 719;  MaxY := 347;  MaxColour := 1;
	      |
		4:	BitsPerPixel := 2;
			BytesPerRow := 80;  FramesPerScreen := 2;
			MaxX := 319;  MaxY := 199;  MaxColour := 3;
	      |
		5:	BitsPerPixel := 2;
			BytesPerRow := 80;  FramesPerScreen := 2;
			MaxX := 319;  MaxY := 199;  MaxColour := 1;
	      |
		6:	BitsPerPixel := 1;
			BytesPerRow := 80;  FramesPerScreen := 2;
			MaxX := 639;  MaxY := 199;  MaxColour := 1;
	      |
		13:	BitsPerPixel := 1;
			BytesPerRow := 40;  FramesPerScreen := 1;
			MaxX := 319;  MaxY := 199;  MaxColour := 15;
	      |
		14:	BitsPerPixel := 1;
			BytesPerRow := 80;  FramesPerScreen := 1;
			MaxX := 639;  MaxY := 199;  MaxColour := 15;
	      |
		15:	BitsPerPixel := 1;
			BytesPerRow := 80;  FramesPerScreen := 1;
			MaxX := 639;  MaxY := 349;  MaxColour := 1;
	      |
		16:	BitsPerPixel := 1;
			BytesPerRow := 80;  FramesPerScreen := 1;
			MaxX := 639;  MaxY := 349;  MaxColour := 15;
	      |
		17:	BitsPerPixel := 1;
			BytesPerRow := 80;  FramesPerScreen := 1;
			MaxX := 639;  MaxY := 479;  MaxColour := 1;
	      |
		18:	BitsPerPixel := 1;
			BytesPerRow := 80;  FramesPerScreen := 1;
			MaxX := 639;  MaxY := 479;  MaxColour := 15;
	      |
		19:	BitsPerPixel := 8;
			BytesPerRow := 320;  FramesPerScreen := 1;
			MaxX := 319;  MaxY := 199;  MaxColour := 255;
	      |
		ELSE
			GetExtendedModeInformation (BytesPerRow, MaxX, MaxY,
					ColourLimit, BitsPerPixel, MultiBank);
			MaxColour := VAL(ColourType, ColourLimit);
			FramesPerScreen := 1;

	    END (*CASE*);

	    VisibleLine := VisibleLine0;
	    IF MaxColour = 15 THEN
		FillProc := EGAFill;

		(* For the 4-plane modes supported in the 16-colour	*)
		(* model, we use set/reset mode as the default.		*)
		OutByte (3CEH, 1);  OutByte (3CFH, 0FH);

	    ELSIF MaxColour >= 255 THEN
		FillProc := Fill256;
		IF MaxColour > 255 THEN
		    VisibleLine := VisibleLine64K;
		END (*IF*);
	    ELSE
		FillProc := Fill0;
	    END (*IF*);

	END (*WITH*);

    END SetMode;

(************************************************************************)

PROCEDURE SetDefaultMode;

    (* Sets the video mode to (our opinion of) the best mode supported	*)
    (* by the hardware.							*)

    BEGIN
	SetMode (DefaultGraphicsMode, TRUE);
    END SetDefaultMode;

(************************************************************************)

PROCEDURE GraphicsOff (ClearScreen: BOOLEAN);

    (* Sets the video mode to a default text mode. *)

    BEGIN
	SetMode (DefaultTextMode, ClearScreen);
    END GraphicsOff;

(************************************************************************)

PROCEDURE GetScreenShape (VAR (*OUT*) xmax, ymax: CARDINAL;
				VAR (*OUT*) maxcolour: ColourType);

    (* Returns the maximum values permitted by the current mode for	*)
    (* x, y, and colour.						*)

    BEGIN
	WITH ModeData DO
	    xmax := MaxX;  ymax := MaxY;
	    maxcolour := MaxColour;
	END (*WITH*);
    END GetScreenShape;

(************************************************************************)
(*			      INITIALISATION				*)
(************************************************************************)

BEGIN
    (* Find out the video adaptor type.	*)

    AdaptorKind := VideoKind();
    FontAddress := Virtual (0FFA6EH);

    (* Work out the "best" modes to use for the available adaptor type.	*)

    CASE AdaptorKind OF
	MDA:		DefaultGraphicsMode := 7;
     |	Hercules:	DefaultGraphicsMode := HercGraphics;
     |	CGA:		DefaultGraphicsMode := 4;
     |	EGA:		DefaultGraphicsMode := 16;
     |	VGA:		DefaultGraphicsMode := 18;
     |	VESA:		DefaultGraphicsMode := 260;
     |	Trident:	DefaultGraphicsMode := 259;
    END (*CASE*);

    IF (AdaptorKind = MDA) OR (AdaptorKind = Hercules) THEN
	DefaultTextMode := 7;
    ELSE
	DefaultTextMode := 3;
    END (*IF*);

    (* Set up default values for the ModeData record.  This is helpful	*)
    (* for debugging: if SetMode is never called, the program will run	*)
    (* in a text mode but the graphics routines will still be working	*)
    (* with plausible values.						*)

    WITH ModeData DO
	mode := 16;  BitsPerPixel := 1;
	BytesPerRow := 80;  FramesPerScreen := 1;
	MaxX := 639;  MaxY := 349;
	MaxColour := 15;
    END (*WITH*);
    FillProc := EGAFill;  VisibleLine := VisibleLine0;
    GetModeData (ScreenSeg, IObase);

END Graphics.
