/* Declarations for `malloc' and friends.
   Copyright 1990, 1991, 1992 Free Software Foundation, Inc.
		  Written May 1989 by Mike Haertel.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; see the file COPYING.LIB.  If
not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA.

   The author may be reached (Email) at the address mike@ai.mit.edu,
   or (US mail) as Mike Haertel c/o Free Software Foundation.  */

#ifndef _MALLOC_H

#define _MALLOC_H	1

/* configuration file */
#include "checker.h"

#ifdef	__cplusplus
extern "C"
{
#endif

#if defined (__cplusplus) || (defined (__STDC__) && __STDC__)
#undef	__P
#define	__P(args)	args
#undef	__ptr_t
#define	__ptr_t		void *
#else /* Not C++ or ANSI C.  */
#undef	__P
#define	__P(args)	()
#undef	const
#define	const
#undef	__ptr_t
#define	__ptr_t		char *
#endif /* C++ or ANSI C.  */

#ifndef	NULL
#define	NULL	0
#endif

#ifdef	__STDC__
#include <stddef.h>
#else
#undef	size_t
#define	size_t		unsigned int
#undef	ptrdiff_t
#define	ptrdiff_t	int
#endif


/* Allocate SIZE bytes of memory.  */
extern __ptr_t malloc __P ((size_t __size));
/* Re-allocate the previously allocated block
   in __ptr_t, making the new block SIZE bytes long.  */
extern __ptr_t realloc __P ((__ptr_t __ptr, size_t __size));
/* Allocate NMEMB elements of SIZE bytes each, all initialized to 0.  */
extern __ptr_t calloc __P ((size_t __nmemb, size_t __size));
/* Free a block allocated by `malloc', `realloc' or `calloc'.  */
extern void free __P ((__ptr_t __ptr));

/* Allocate SIZE bytes allocated to ALIGNMENT bytes.  */
extern __ptr_t memalign __P ((size_t __alignment, size_t __size));

/* Allocate SIZE bytes on a page boundary.  */
extern __ptr_t valloc __P ((size_t __size));


#ifdef _MALLOC_INTERNAL

#include <stdio.h>		/* Harmless, gets __GNU_LIBRARY__ defined.  */

#include <chkrlib.h>

#if 0
#if defined(__GNU_LIBRARY__) || defined(STDC_HEADERS) || defined(USG)
#include <string.h>
#else
#ifndef memset
#define	memset(s, zero, n)	bzero ((s), (n))
#endif
#ifndef memcpy
#define	memcpy(d, s, n)		bcopy ((s), (d), (n))
#endif
#endif
#endif

#if	defined(__GNU_LIBRARY__) || defined(__STDC__)
#include <limits.h>
#else
#define	CHAR_BIT	8
#endif

/* The allocator divides the heap into blocks of fixed size; large
   requests receive one or more whole blocks, and small requests
   receive a fragment of a block.  Fragment sizes are powers of two,
   and all fragments of a block are the same size.  When all the
   fragments in a block have been freed, the block itself is freed.  */
#define INT_BIT		(CHAR_BIT * sizeof(int))
#define BLOCKLOG	(INT_BIT > 16 ? 12 : 9)
#define BLOCKSIZE	(1 << BLOCKLOG)
#define BLOCKIFY(SIZE)	(((SIZE) + BLOCKSIZE - 1) / BLOCKSIZE)

/* Determine the amount of memory spanned by the initial heap table
   (not an absolute limit).  */
#define HEAP		(INT_BIT > 16 ? 4194304 : 65536)

/* Number of contiguous free blocks allowed to build up at the end of
   memory before they will be returned to the system.  */
#define FINAL_FREE_BLOCKS	16	/* 8 */

/* Data structure giving per-block information.  */
typedef union
  {
    /* Heap information for a busy block.  */
    struct
      {
	union
	  {
	    struct
	      {
		size_t nfree;	/* Free fragments in a fragmented block.  */
		size_t first;	/* First free fragment of the block.  */
	      } frag;
	    /* Size (in blocks) of a large cluster.  */
	    size_t size;
	  } info;
	/* Zero for a large block, or positive giving the
	   logarithm to the base two of the fragment size.  */	  
	int type;  
      } busy;
    /* Heap information for a free block
       (that may be the first of a free cluster).  */
    struct
      {
	size_t size;		/* Size (in blocks) of a free cluster.  */
	size_t next;		/* Index of next free cluster.  */
	size_t prev;		/* Index of previous free cluster.  */
      } free;
#if CHECKER     
    struct
      {
        size_t pad1[3];		/* pad to hid 'free' & 'busy' */
        char   state;		/* state: free or busy */
        char   garbage;		/* For a garbage collector */
        short int delay;	/* For adding a delay before allocating */
      } status;
#endif /* CHECKER */
  } malloc_info;

#if CHECKER
/* define .status.state value */
#define MDFREE 0
#define MDHEAD 1
#define MDCORP 2
#define MDTAIL 3
#define MDINTERN 4
#define MDCHECK 128 /* a flag */
#endif /* CHECKER */
  
/* Pointer to first block of the heap.  */
extern char *_heapbase;

/* Table indexed by block number giving per-block information.  */
extern malloc_info *_heapinfo;

/* Address to block number and vice versa.  */
#define BLOCK(A)	(((char *) (A) - _heapbase) / BLOCKSIZE + 1)
#define ADDRESS(B)	((__ptr_t) (((B) - 1) * BLOCKSIZE + _heapbase))

/* Current search index for the heap table.  */
extern size_t _heapindex;

/* Limit of valid info table indices.  */
extern size_t _heaplimit;

/* Doubly linked lists of free fragments.  */
struct list
  {
    struct list *next;
    struct list *prev;
  };

/* Free list headers for each fragment size.  */
extern struct list _fraghead[];

/* List of blocks allocated with `memalign' (or `valloc').  */
struct alignlist
  {
    struct alignlist *next;
    __ptr_t aligned;		/* The address that memaligned returned.  */
    __ptr_t exact;		/* The address that malloc returned.  */
  };
extern struct alignlist *_aligned_blocks;

#ifdef CHKR_MSTAT
/* Instrumentation.  */
extern size_t _chunks_used;
extern size_t _bytes_used;
extern size_t _chunks_free;
extern size_t _bytes_free;
#endif /* CHKR_MSTAT */

/* Internal version of `free' used in `morecore' (malloc.c). */
extern void _free_internal __P ((__ptr_t __ptr));

/* Intrenal version of `malloc' see malloc.c */
extern __ptr_t _malloc_internal __P ((size_t __size)); 

#ifdef CHKR_GARBAGE
/* size of red zones */
extern unsigned int be_red_zone;	/* bytes, but must long */
extern unsigned int af_red_zone;

/* structure at the top of the red zone */
struct hdr_red_zone
{
 size_t real_size;	/* size */
 char   garbage_t;	/* type of garbage: sure, possibly, not */
 char   pad[2];
 char   g_flag;		/* how to interpret g_link */
 union 
 {
   struct hdr_red_zone *g_parent;	/* common group */
   unsigned int	 g_number;		/* number of 'brother' */
 } g_link;	/* looking for a better name */
 size_t full_size;	/* the size allocated by malloc (block or frag) */
 unsigned long magic;	/* magic number */
};

/* magic number in magic (Yes, I'm french :-) */
#define RED_ZONE_MAGIC 0xcafe0cfc

/* Values of garbage_t */
#define POINT_NOT   0	/* no pointer on it */
#define POINT_SURE  1	/* a pointer on the begin */
#define POINT_MAYBE 2	/* a pointer inside */
#define POINT_LEAK  3   /* This is a leak */
#define POINT_SEEN  4	/* already seen (for new zones) */
#define POINT_JUST_SEEN  5	/* to avoid infinitely recursive calls */

/* Values of g_flag */
#define G_ALONE 0	/* no brothers */
#define G_CHILD 2	/* a child. g_group points to its parents */
#define G_PARENT 3	/* a parent. g_number is the number of childs */

/* Defined in garbage.c */
void foreach_busy __P ((void (*func)(__ptr_t ptr)));
__ptr_t chkr_address __P ((__ptr_t ptr));
#endif /* CHKR_GARBAGE */
#ifdef CHKR_SAVESTACK
/* Defined in sysdep.c */
void chkr_save_stack __P((__ptr_t *ptr, int forget, int num));
void chkr_initialize();
extern char *chkr_prog_path;
/* Defined in find_exec.c */
char *chkr_find_executable __P ((char *file));
/* Defined in symtab.c */
void chkr_use_symtab __P ((void (*func)(int status)));
void chkr_show_addr __P ((__ptr_t *ptr));
#endif /* CHKR_SAVESTACK */

#ifdef CHKR_USE_BITMAP
#define CHKR_UN	0	/* Can't neither read nor write */
#define CHKR_RO	1	/* Read only */
#define CHKR_WO	2	/* Write only */
#define CHKR_RW 3	/* Read and write */
#define CHKR_TW 6	/* test for write, but no evloution */
#ifdef CHKR_HEAPBITMAP
extern __ptr_t chkr_heap_begin;
extern unsigned char *chkr_heap_bitmap;
#endif /* CHKR_HEAPBITMAP */
#ifdef CHKR_DATABITMAP
extern unsigned char *chkr_data_bitmap;
#endif /* CHKR_DATABITMAP */
#ifdef CHKR_STACKBITMAP
extern unsigned char *chkr_stack_bitmap;
#endif /* CHKR_STACKBITMAP */
extern int bm_round;		/* number of bytes per byte of bitmap -1 */
extern int bm_log_size;
void chkr_set_right(__ptr_t ptr, int len, int right);
void chkr_check_addr(__ptr_t ptr, int len, int right);
void chkr_check_str(__ptr_t ptr, int right);
#endif /* CHKR_USE_BITMAP */
#endif /* _MALLOC_INTERNAL.  */

#if CHECKER
/* Fonction called in case of error */
void chkr_abort();
void chkr_garbage_detector();
void chkr_evol_detector();
#ifdef CHKR_DEBUG
void __chkr_dump_frag();
void __chkr_dump_free();
void __chkr_dump_block __P((const size_t block));
void __chkr_check_intern();
void __chkr_dump_busy();
#endif /* CHKR_DEBUG */
extern int chkr_errno;	/* error number */
extern int chkr_show_err_flag;	/* flag: print error when detect */
extern const char * const chkr_errlist[];
void chkr_perror();
#endif /* CHECKER */

/* Underlying allocation function; successive calls should
   return contiguous pieces of memory.  */
extern __ptr_t (*__morecore) __P ((ptrdiff_t __size));

/* Default value of `__morecore'.  */
extern __ptr_t __default_morecore __P ((ptrdiff_t __size));

/* Nonzero if `malloc' has been called and done its initialization.  */
extern int __malloc_initialized;

#ifdef CHKR_HOOK
/* Hooks for debugging versions.  */
extern void (*__free_hook) __P ((__ptr_t __ptr));
extern __ptr_t (*__malloc_hook) __P ((size_t __size));
extern __ptr_t (*__realloc_hook) __P ((__ptr_t __ptr, size_t __size));

/* Activate a standard collection of debugging hooks.  */
extern void mcheck __P ((void (*__func) __P ((void))));

/* Activate a standard collection of tracing hooks.  */
extern void mtrace __P ((void));
#endif /* CHKR_HOOK */

#ifdef CHKR_MSTAT
/* Statistics available to the user.  */
struct mstats
  {
    size_t bytes_total;		/* Total size of the heap. */
    size_t chunks_used;		/* Chunks allocated by the user. */
    size_t bytes_used;		/* Byte total of user-allocated chunks. */
    size_t chunks_free;		/* Chunks in the free list. */
    size_t bytes_free;		/* Byte total of chunks in the free list. */
  };

/* Pick up the current statistics. */
extern struct mstats mstats __P ((void));
#endif /* CHKR_MSTAT */

#ifdef	__cplusplus
}
#endif

#endif /* malloc.h  */
