(*
$File: Compiler/PatBindings.sml $
$Date: 1992/09/17 14:18:02 $
$Revision: 1.1 $
$Locker:  $
*)

(*$PatBindings:
	LAB VAR RESIDENT DEC_GRAMMAR COMPILER_ENV GRAMMAR_INFO LVARS PRETTYPRINT
	FINMAP CRASH PAT_BINDINGS
 *)

functor PatBindings(structure Lab: LAB
		    structure Var: VAR
		    structure ResIdent: RESIDENT
		      sharing type ResIdent.longvar = Var.longvar

		    structure Grammar: DEC_GRAMMAR
		      sharing type Grammar.lab = Lab.lab
			  and type Grammar.id = Var.var
			  and type Grammar.longid = ResIdent.longid

		    structure CompilerEnv: COMPILER_ENV
		      sharing type CompilerEnv.var = Var.var

		    structure GrammarInfo: GRAMMAR_INFO
		      sharing type Grammar.info
				   = GrammarInfo.PostElabGrammarInfo

		    structure Lvars: LVARS
		      sharing type CompilerEnv.lvar = Lvars.lvar

		    structure PP: PRETTYPRINT
		      sharing type CompilerEnv.StringTree = PP.StringTree

		    structure FinMap: FINMAP
		      sharing type FinMap.StringTree = PP.StringTree

		    structure Crash: CRASH
		   ): PAT_BINDINGS =
  struct
    open Grammar ResIdent CompilerEnv
    type var = Var.var
    type lvar = Lvars.lvar
    type RuleNum = int
    type TypeInfo = GrammarInfo.TypeInfo
    type (''a, 'b) map = (''a, 'b) FinMap.map
    infix plus
    val (op plus) = CompilerEnv.plus

    datatype BindingTree =
        TUPLEbtree of (lab, (TypeInfo * lvar * BindingTree)) map
      | CONbtree   of {child: BindingTree, childLvar: lvar}
      | EXCONbtree of {child: BindingTree, childLvar: lvar}
      | NILbtree

   (*A couple of utility functions not needed elsewhere.*)

    fun zip3(hd :: tl, hd' :: tl', hd'' :: tl'') =
	  (hd, hd', hd'') :: zip3(tl, tl', tl'')
      | zip3(nil, nil, nil) = nil
      | zip3 _ = Crash.impossible "zip3"

    local
      fun unzip3'((x, y, z) :: rest, xs, ys, zs) =
	    unzip3'(rest, x :: xs, y :: ys, z :: zs)
	| unzip3'(nil, xs, ys, zs) = (xs, ys, zs)
    in
      fun unzip3 triples = unzip3'(rev triples, nil, nil, nil)
    end

    fun patBindings(root, pat): (BindingTree * CEnv) =
      case pat
	of ATPATpat(_, atpat) =>
	     atpatBindings(root, atpat)

	 | CONSpat(_, _, atpat) =>
	     let
	       val childLv = Lvars.newLvar()
	       val (bt, env) = atpatBindings(childLv, atpat)
	     in
	       (CONbtree{child=bt, childLvar=childLv}, env)
	     end

	| TYPEDpat(_, pat, _) =>
	    patBindings(root, pat)

        | LAYEREDpat(_, OP_OPT(id, _), _, pat) =>
	    let
	      val bind = declareVar(id, root, emptyCEnv)
	      val (bt, env) = patBindings(root, pat)
	    in
	      (bt, env plus bind)
	    end

	| UNRES_INFIXpat _ =>
	    Crash.impossible "patBindings(UNRES_INFIX)"

    and atpatBindings(root, atpat): (BindingTree * CEnv) =
      case atpat
	of WILDCARDatpat _ =>
	     (NILbtree, emptyCEnv)

	 | SCONatpat _ =>
	     (NILbtree, emptyCEnv)

	 | LONGIDatpat(_, OP_OPT(longid, _)) =>
	     (NILbtree, case longid
			  of LONGVAR longvar =>
			       (case Var.decompose longvar
				  of (nil, var) =>
				       declareVar(var, root, emptyCEnv)
				   | _ => Crash.impossible "atpatBindings"
			       )

			   | LONGCON _   => emptyCEnv
			   | LONGEXCON _ => emptyCEnv
	     )

	 | RECORDatpat(_, patrowOpt) =>
	     let
	       val (labs, infosXlvarsXpats) =
		 let
		   fun f None = nil
		     | f (Some(PATROW(i, lab, pat, rest))) =
					(* We must always have TypeInfo here
					   for labels... *)
		           let
			     val ti =
			       case GrammarInfo.getPostElabTypeInfo i
				 of Some ti => ti
				  | None =>
				      Crash.impossible
				        "PatBindings.atpatBindings(ti)"
			   in
			     (lab, (ti, Lvars.newLvar(), pat)) :: f rest
			   end
		     | f (Some(DOTDOTDOT _)) = nil
		 in
		   ListPair.unzip(f patrowOpt)
		 end

	       val (infos, lvars, pats) = unzip3 infosXlvarsXpats

	       val (trees, envs) =
		 ListPair.unzip(map patBindings (ListPair.zip(lvars, pats)))

	       val L: (lab * (TypeInfo * lvar * BindingTree)) list =
		 ListPair.zip(labs, zip3(infos, lvars, trees))

	       val map =
		 List.foldL
		   (fn (lab: (*eqtype*) Lab.lab, infoXlvarXbtree) =>
		      fn map => FinMap.add(lab, infoXlvarXbtree, map)
		   ) FinMap.empty L

	       val totalEnv = List.foldL (General.curry op plus) emptyCEnv envs
	     in
	       (TUPLEbtree map, totalEnv)
	     end

	 | PARatpat(_, pat) =>
	     patBindings(root, pat)


    type StringTree = PP.StringTree

    fun layoutPatBindings(tree: BindingTree, env: CEnv) =
      let
	fun layoutBTree tree =
	  case tree
	    of TUPLEbtree map =>
	         FinMap.layoutMap
		   {start="TUPLEbtree{", eq=" -> ", sep="; ", finish="}"}
		   (PP.layoutAtom Lab.pr_Lab)
		   (fn (_, lv, t) =>
		      PP.NODE{start="(" ^ Lvars.pr_lvar lv ^ ", ",
			      finish=")", indent=3, childsep=PP.NONE,
			      children=[layoutBTree t]
			     }
		   )
		   map

	     | CONbtree{child, childLvar} =>
	         PP.NODE{start="CONbtree(" ^ Lvars.pr_lvar childLvar ^ ": ",
			 finish=")", indent=3, childsep=PP.NONE,
			 children=[layoutBTree child]
			}

	     | EXCONbtree{child, childLvar} =>
	         PP.NODE{start="EXCONbtree(" ^ Lvars.pr_lvar childLvar ^ ": ",
			 finish=")", indent=3, childsep=PP.NONE,
			 children=[layoutBTree child]
			}

	     | NILbtree => PP.LEAF "NILbtree"
      in
	PP.NODE{start="Pat Bindings: ", finish="",
		indent=3, childsep=PP.RIGHT "; ",
		children=[layoutBTree tree, layoutCEnv env]
	       }
      end
  end;
