/*
 *   Cloak of Darkness - a simple demonstration of Interactive Fiction
 *     This version for TADS written by Dan Shiovitz on 23Sep99
 *       adapted from the version for Inform written by Roger Firth on 17Sep99
 */

#include <adv.t>
#include <std.t>

// First let's modify the version so the initial title and title given
// by the 'VERSION' command are correct:
modify version
  sdesc = "\(Cloak of Darkness\)\n
           A basic IF demonstration."
  // \( and \) turn boldface on/off
  // \n is a newline
  // Note that we can hit return in the middle of a string and it turns into
  // a single space in the output
;

// And let's set the max possible number of points to 2.
modify global
  maxscore = 2
;


// The startroom is an instance of the basic room class, because it's a
// basic room. It is also the room in which the player will start out
// (hence the name).
startroom: room
  // sdesc = 'short description' and ldesc = 'long description', of course
  sdesc = "Foyer of the Opera House"
  ldesc = "You are standing in a spacious hall, splendidly decorated in red
           and gold, with glittering chandeliers overhead. The entrance from
           the street is to the north, and there are doorways south and west."

  // south and west lead to other rooms
  south = bar
  west = cloakroom

  // north instead prints a message and returns nil, to show that this is
  // not an exit that moves the player anyplace when it is taken
  north =
  {
    "You've only just arrived, and besides, the weather outside
     seems to be getting worse. ";
    return nil;
  }
;

// Another basic room, although this one has an object in it
cloakroom: room
  sdesc = "Cloakroom"
  ldesc = "The walls of this small room were clearly once lined with hooks,
           though now only one remains. The exit is a door to the east."
  east = startroom
;

// Here is the one item, a hook on the wall.
// It is of class item, meaning it is a normal item with all the default
// responses to verbs (eat, pull, push, etc) defined. But it is also of
// class fixeditem, meaning it is fixed in place and cannot be moved.
// The fixeditem class therefore defines special responses to "take" "put in"
// etc.
// The fixeditem class is also commonly used for items which, like this,
// we do not wish to be listed in the room description, so it sets a
// special do-not-list-me flag. But that's handled automatically, so
// we don't have to worry about it here.
// Finally, it is of class surface, meaning you can put things on it, and
// it describes its contents in the room description.

hook: fixeditem, surface, item
  sdesc = "small brass hook"
  // whereas for the other items the ldesc is just a string, here it is
  // a method that prints different strings depending on whether the cloak
  // is on the hook or not
  ldesc =
  {
    "It's just a small brass hook, ";

    // ok so does the cloak object have us as its location?
    if (cloak.isIn(self))
      "with a cloak hanging on it. ";
    else
      "screwed to the wall. ";
  }

  // here are some keywords by which this item can be referred to
  noun = 'hook' 'peg'
  adjective = 'small' 'brass'

  // and here's where the item is located in the game
  location = cloakroom
;

// Also, we want to allow the syntax "HANG CLOAK ON HOOK" as well as
// "PUT CLOAK ON HOOK". The easiest way to do this is just to modify
// putVerb so "hang" is a synonym for "put".
modify putVerb
  verb = 'hang'
;

// Ok, here's the last room. This is kind of a special room,
// since it's dark if you're wearing the cloak and light if you aren't.
// Luckily, TADS has a room-which-is-dark class defined already, namely
// darkroom. And this class has a method, lightsOn, which determines if the
// lights should be on or not. So:
bar: darkroom
  lightsOn =
  {
    // As you recall, the lights are on only if the cloak is not in this
    // room.
    if (not cloak.isIn(self)) return true;
    else                      return nil; // "nil" is TADSspeak for "false"
  }

  sdesc = "Foyer bar"
  ldesc = "The bar, much rougher than you'd have guessed after the opulence
           of the foyer to the north, is completely empty. There seems to
           be some sort of message scrawled in the sawdust on the floor."
  north = startroom

  // Now, if you wander around in this room you'll scuff the message. You also
  // scuff the message if you do any other actions. We use the
  // "roomCheck" method to trap the appropriate verbs. Oh, and you
  // don't scuff the message if the lights are on in here.
  roomCheck(verb) =
  {
    if (not self.islit) // if the lights are off, then ...
    {
      // If the verb is a travelVerb (eg, north, south, east, west ...)
      // but it's not north, then we print a message.
      if (isclass(verb, travelVerb))
      {
        if (verb <> nVerb) // if it's not north
        {
          "Blundering around in the dark isn't a good idea!";
          message.number += 2;  // and we increment this counter

          // and lastly we return nil, which stops further processing of
          // the verb so you don't do the normal thing.
          return nil;
        }
      }

      // we also want to check all other verbs which are not sysverbs.
      // sysverbs are those "meta" verbs like save, restore, undo,
      // that usually shouldn't be trapped by in-game things.
      else if (not isclass(verb, sysverb))
      {
        "In the dark? You could easily disturb something!";
        message.number += 1; // add to counter, but not as much

        // and again, we return nil
        return nil;
      }
    }

    // if we didn't return nil before, return true here, to let the
    // verb do its normal action.
    return true;
  }
;


// The message in the floor probably shouldn't be picked up, so let's
// make it a fixeditem also. If you look at it, you win/lose the game,
// depending on how scuffled you made it. Also, the message should be
// readable by typing "read message" as well as by looking at it. The
// simplest way to do this is to make it part of the readable class.
// This class by default makes doing "read message" the same as
// doing "examine message" which is just what we want.
message: fixeditem, readable
  number = 0 // the message's number counter will start at 0
             // and increase the more the player walks around in the bar

  sdesc = "scrawled message"

  // again, this is a method. It'll call a function that we'll write later
  ldesc =
  {
    if (self.number < 2)
    {
      incscore(1); // you get a point for not messing it up
      "The message, neatly marked in the sawdust, reads...";
      endGame(true); // this function will print "you have won" and quit.
    }
    else // uh-oh! it's messed up!
    {
      "The message has been carelessly trampled, making it difficult to read.
       You can just distinguish the words...";
      endGame(nil); // this function will print "you have lost" and quit.
    }
  }
  noun = 'message' 'sawdust' 'floor' 'dust'
  adjective = 'saw'
  location = bar
;

// The cloak is an article of clothing, so it can be worn. When you put
// it on the hook, you get a point, if you haven't done it before.
cloak: clothingItem
  sdesc = "velvet cloak"
  ldesc = "A handsome cloak, of velvet trimmed with satin, and slightly
           spattered with raindrops. Its blackness is so deep that it
           almost seems to suck light from the room.",
  noun = 'cloak'
  adjective = 'handsome' 'dark' 'black' 'velvet' 'satin' 'cloak'

  location = Me // Me is the player object, so the cloak starts out in
                // the player's inventory
  isworn = true // and the isworn flag is true, so it starts out being worn

  hooked = nil // this is the flag that says if it's been put on the hook or
               // not

  // this is the method called to put the cloak on something
  // doPutOn = this is being used as the direct object of the PutOn verb.
  // the actor and indirect object are passed as arguments.
  // We want to check if the indirect object is the hook, and if so,
  // award a point, as long as we haven't already done so
  doPutOn(actor, iobj) =
  {
    if (iobj = hook and not hooked)
    {
      incscore(1);
      hooked := true; // by setting hooked to true, we make sure this
                      // will not happen more than once
    }
    pass doPutOn; // the pass statement means "now do the rest of the normal
                  // stuff" which in this case means putting it onto the hook.
  }


  // the verDoDrop method is also a method called when this object is the
  // direct object of the drop verb, but the "ver" on the front means it
  // is a "verify" method. That is, it is called to see if the player is
  // allowed to *try* and drop the cloak, unlike doDrop, which would be
  // called when the player actually goes ahead and drops it.
  // But we don't want to ever let them drop the cloak, so we'll
  // print an error message. This will prevent the verDoDrop method from
  // succeeding, so the player will never be allowed to drop it.

  verDoDrop(actor) = // "drop" has no possible indirect object
  {
    if (actor.location = cloakroom)
      "Why put it on the floor when there's a perfectly good hook!";
    else
      "This isn't the best place to leave a smart cloak lying around.";
  }
;

// This is the function that gets called at the very start of the game:
replace init: function
{
  // Some introductory text:
  "\b\bHurrying through the rainswept November night, you're glad to see the
    bright lights of the Opera House. It's surprising that there aren't more
    people about but, hey, what do you expect in a cheap demo game...?\b\b";

  // The name of the game:
  version.sdesc;
  "\b";

  // Move the player to the foyer:
  Me.location := startroom;

  // Have a look around:
  Me.location.lookAround(true);

  // And make a note that we've see the room now:
  Me.location.isseen := true;

  // Lastly, start off the turn-counter and the status line:
  setdaemon(turncount, nil);
  scoreStatus(0,0);
}


// And lastly here's our end-the-game thing:
endGame: function(won)
{
  if (won)
    "\b*** You have won ***\b";
  else
    "\b*** You have lost ***\b";

  scoreRank(); // give the final score

  "[Hit any key to continue]\n";

  // Now wait for the player to enter a key:
  inputkey();

  // and then quit.
  quit();
}
