(**
 * *DS* An Interface for the Ispell spell checker.
 **)

unit IspellInterface;

interface

uses
  classes, SysUtils, Windows, Dialogs;

const
 LF: string = #13#10;

 // Ispell return values word known
 ISPELL_OK = 1;
 ISPELL_ROOT = 2;
 ISPELL_COMPOUND = 3;
 // Ispell return values word unknown
 ISPELL_MISS = 4;
 ISPELL_GUESS = 5;
 ISPELL_NONE = 6;

 BUFFER_SIZE = 4096;

 // used to parse Ispell's output line
 Delimiters = [',', ' ', ':', #13, #10];
 LineDelimiters = [#13, #10];
 // Ispell allways begins output line with one of these signs
 IspellSigns =['*', '+', '-', '&', '?', '#'];


type
  IspellFirst = record
    pos: integer;  // position of first misspelled word
    len: integer;  // lenght of first misspelled word
  end;

  //
  // Ispell.CheckWord returns this object.
  // value: Shows if Ispell knows the word.
  // list: List of alternative words.
  //
  IspellResult = class
    value: integer;
    list: TStringList;
  end;

  //
  // Interface for the Ispell spell checker
  //
  Ispell = class
  public
    constructor Create (command: String);
    destructor Free;

    // searches in a string for the first misspelled word
    // returns true if a misspelled word is found
    // position and lenght of the misspelled word are returned in first.
    function FindFirst (strLine: String; var first: IspellFirst): boolean;

    function CheckWord (strWord: String): IspellResult;

    // Add a word to the dictionary
    procedure AddWord (strWord: String);

    // Accept a word, but don't put it into the dictionary
    procedure AcceptWord (strWord: String);

  protected
    // handles for reading and writing on the TeXShell's side
    childPipeOutRd: THandle;
    //childPipeInWrDup : THandle;
    childPipeInWr: THandle;

    // Parses an Ispell output line and returns Ispell's
    // list of alternatives for a word.
    function ParseInput (line: PChar): TStringList;

    // parses an Ispell output line and returns the position of
    // the misspelled word and it's length in first
    // If the output signals a correct word false is returned.
    function GetPosition (line: PChar; var first: IspellFirst) : boolean;
  end;


implementation


constructor Ispell.Create (command: String);
var
  startupInfo: TStartupInfo;
  processInfo: TProcessInformation;
  securityAttr: TSecurityAttributes;
  // current standard input and output handles of the TeXShell
  //mainPipeIn: THandle;
  //mainPipeOut: THandle;

  // handles of the child process
  childPipeIn: THandle;
  childPipeOut: THandle;
  //childPipeInWr : THandle;
  success: Boolean;
  cBytes: DWORD;
  inBuffer:  array[0..BUFFER_SIZE] of Char;
begin
  //mainPipeIn := GetStdHandle (STD_INPUT_HANDLE);
  //mainPipeOut := GetStdHandle (STD_OUTPUT_HANDLE);

  with securityAttr do
  begin
    nLength := SizeOf (TSecurityAttributes);
    lpSecurityDescriptor := nil;
    bInheritHandle := true;
  end;

  // create pipe for child's standard output
  success := CreatePipe (childPipeOutRd, childPipeOut, Addr(securityAttr), 0);
  SetHandleInformation( childPipeOutRd, HANDLE_FLAG_INHERIT, 0 );

  // create pipe for child's standard input
  success := CreatePipe (childPipeIn, childPipeInWr, Addr(securityAttr), 0);
  SetHandleInformation( childPipeInWr, HANDLE_FLAG_INHERIT, 0 );



(*
  success := SetStdHandle (STD_INPUT_HANDLE, childPipeIn);
  success := SetStdHandle (STD_OUTPUT_HANDLE, childPipeOut);

  success := DuplicateHandle(GetCurrentProcess(), childPipeInWr,
        GetCurrentProcess(), Addr(childPipeInWrDup), 0,
        FALSE,
        DUPLICATE_SAME_ACCESS);

  success := CloseHandle (childPipeInWr);
*)

  // Create Ispell process

  with startupInfo do
  begin
    cb := SizeOf (TStartupInfo);
    lpReserved := nil;
    lpDesktop := nil ;
    lpTitle := nil;
    dwX := 0;
    dwY := 0;
    dwXSize := 0;
    dwYSize := 0;
    dwXCountChars := 0;
    dwYCountChars := 0;
    dwFillAttribute := 0;
    //dwFlags := 0;
    dwFlags := STARTF_USESTDHANDLES;
    wShowWindow := 0;
    cbReserved2 := 0;
    lpReserved2:= nil;
    //hStdInput := 0;
    hStdInput := childPipeIn;
    //hStdOutput:= 0;
    hStdOutput:= childPipeOut;
    //hStdError := 0;
    hStdError := childPipeOut;
  end;

  success := CreateProcess (nil,
                 PChar(command),
                 nil,
                 nil,
                 true,
                 DETACHED_PROCESS,
                 //0,
                 nil,
                 nil,
                 startupInfo,
                 processInfo);

  if success = false then
  begin
    //CloseHandle (childPipeInWrDup);
    CloseHandle (childPipeInWr);
    CloseHandle (childPipeOutRd);
    raise EInOutError.Create ('Could not connect to Spell Checker!');
  end;

  //success := SetStdHandle (STD_INPUT_HANDLE, mainPipeIn);
  //success := SetStdHandle (STD_OUTPUT_HANDLE, mainPipeOut);

  CloseHandle(processInfo.hThread);
  CloseHandle(processInfo.hProcess);

  // read first output (Ispell version identification)
  success := ReadFile (childPipeOutRd ,inBuffer, BUFFER_SIZE, cBytes, nil);
  //OemToCharBuff (inBuffer, inBuffer, cBytes);
  while (inBuffer[cBytes-1] <> #10) AND (success = true) do
  begin
    success := ReadFile (childPipeOutRd ,inBuffer, BUFFER_SIZE, cBytes, nil);
    //OemToCharBuff (inBuffer, inBuffer, cBytes);
  end

end;


destructor Ispell.Free;
begin
  CloseHandle (childPipeInWr);
  CloseHandle (childPipeOutRd);
end;


function Ispell.ParseInput (line: PChar): TStringList;
var
  pivot,
  endPivot: PChar;
  list: TStringList;
  workBuffer: array[0..BUFFER_SIZE] of Char;
begin
  list := TStringList.create;

  pivot := line;

  // move pivot one position after :
  while pivot^ <> ':' do Inc (pivot);
  Inc (pivot);

  while (true) do
  begin
    // find word
    while pivot^ in Delimiters do Inc (pivot);
    endPivot := pivot;
    while not (endPivot^ in Delimiters) do Inc (endPivot);

    // copy word into a string and add it to list
    StrLCopy (workBuffer, pivot, endPivot-pivot);
    list.add (string(workBuffer));

    list.count;
    list.strings[0];

    if endPivot^ in LineDelimiters then break;
    pivot := endPivot;
  end;
  result := list;
end;


// parses an Ispell output line and returns the position of
// the misspelled word and it's length in the variable first
// If the output signals a correct word false is returned.
function Ispell.GetPosition (line: PChar; var first: IspellFirst) : boolean;
var
  pivot,
  endPivot: PChar;
  spaces: integer;
  i: integer;
  errCode,
  value: integer;
  workBuffer: array[0..BUFFER_SIZE] of Char;
begin
  result := false;

  // determine how many space characters are between the
  // misspelled word and the offset
  case line[0] of
    '&' : spaces := 2;
    '?' : spaces := 2;
    '#' : spaces := 1;
  else
    Exit;
  end;

  // find length of misspelled word
  pivot := line + 2; // beginning of misspelled Word
  endPivot := pivot;
  while endPivot^ <> ' ' do Inc (endPivot);
  first.len := endPivot - pivot;

  pivot := endPivot;

  // jump over spaces
  for i := 1 to spaces do
  begin
    // move pivot one position after :
    while pivot^ <> ' ' do Inc (pivot);
    Inc (pivot);
  end;

  // find position
  // while pivot^ in Delimiters do Inc (pivot);
  endPivot := pivot;
  while not (endPivot^ in Delimiters) do Inc (endPivot);

  // convert word into a number and return
  StrLCopy (workBuffer, pivot, endPivot-pivot);
  Val (String(workBuffer), value, errCode);
  if errCode > 0 then
    result := false
  else
  begin
    first.pos := value;
    result := true;
  end;
end;


function Ispell.CheckWord (strWord: String): IspellResult;
var
  cBytes: DWORD;
  bytesRead: DWORD;
  success: boolean;
  strLine: string;
  wbMark: PChar;
  outBuffer: array[0..BUFFER_SIZE] of Char;
  inBuffer:  array[0..BUFFER_SIZE] of Char;
  workBuffer: array[0..BUFFER_SIZE] of Char;
begin
  strLine := '^' + strWord + LF;
  StrPLCopy (outBuffer, strLine, BUFFER_SIZE);
  //CharToOem (outBuffer, outBuffer);
  success := WriteFile (childPipeInWr, outBuffer, Length(strLine), cBytes, nil);

  bytesRead := 0;
  wbMark := workBuffer;

  repeat
    success := ReadFile (childPipeOutRd ,inBuffer, BUFFER_SIZE, cBytes, nil);
    //OemToCharBuff (inBuffer, inBuffer, cBytes);
    StrPLCopy (wbMark, inBuffer, cBytes);
    Inc (wbMark, cBytes);
    Inc (bytesRead, cBytes);
  until ((bytesRead >= 4)
        and  (workBuffer[bytesRead-1] = #10)
        and  (workBuffer[bytesRead-3] = #10))
    or   (success = false)
    // in math mode Ispell returns just a line break
    or   ((workBuffer[0] = #13)
           and  (workBuffer[1] = #10));

  result := IspellResult.Create;

  case workBuffer[0] of
    '*' : result.value := ISPELL_OK;
    '+' : result.value := ISPELL_ROOT;
    '-' : result.value := ISPELL_COMPOUND;
    '&' : begin
            result.value := ISPELL_MISS;
            result.list := ParseInput (workBuffer);
          end;
    '?' : result.value := ISPELL_GUESS;
    '#' : result.value := ISPELL_NONE;
    else result.value := 0;
  end;
end;


procedure Ispell.AddWord (strWord: String);
var
  cBytes: DWORD;
  success: boolean;
  strLine: string;
  outBuffer: array[0..BUFFER_SIZE] of Char;
begin
  strLine := '*' + strWord + LF;
  StrPLCopy (outBuffer, strLine, BUFFER_SIZE);
  //CharToOem (outBuffer, outBuffer);
  success := WriteFile (childPipeInWr, outBuffer, Length(strLine), cBytes, nil);
end;


procedure Ispell.AcceptWord (strWord: String);
var
  cBytes: DWORD;
  success: boolean;
  strLine: string;
  outBuffer: array[0..BUFFER_SIZE] of Char;
begin
  strLine := '@' + strWord + LF;
  StrPLCopy (outBuffer, strLine, BUFFER_SIZE);
  //CharToOem (outBuffer, outBuffer);
  success := WriteFile (childPipeInWr, outBuffer, Length(strLine), cBytes, nil);
end;


// searches in a string for the first misspelled word
// returns true if a misspelled word is found
// position and lenght of the misspelled word are returned in first.
function Ispell.FindFirst (strLine: String; var first: IspellFirst): boolean;
var
  cBytes: DWORD;
  success: boolean;
  pos : integer;
  wbMark: PChar; // mark workbuffer position
  ibEnd,
  iba,
  ibb: PChar; // mark inBuffer position
  isEmptyLine: boolean;
  misspelled: boolean;
  found: IspellFirst;
  outBuffer: array[0..BUFFER_SIZE] of Char;
  inBuffer:  array[0..BUFFER_SIZE] of Char;
  workBuffer: array[0..BUFFER_SIZE] of Char;
begin
  result := false;

  // write line out to Ispell
  strLine := '^' + strLine + LF;
  StrPLCopy (outBuffer, strLine, BUFFER_SIZE);
  //CharToOem (outBuffer, outBuffer);
  success := WriteFile (childPipeInWr, outBuffer, Length(strLine), cBytes, nil);

  wbMark := workBuffer;
  iba := inBuffer;
  ibb := inBuffer;
  isEmptyLine := false;

  // parse output from Ispell until an empty line occurs
  // that signals the end of Ispell's output
  while not isEmptyLine do
  begin
    // read Ispell output and parse lines
    success := ReadFile (childPipeOutRd, inBuffer, BUFFER_SIZE, cBytes, nil);
    //OemToCharBuff (inBuffer, inBuffer, cBytes);
    ibEnd := inBuffer + cBytes - 1;
    iba := inBuffer;
    ibb := inBuffer;

    // extract lines from inBuffer and copy them to work buffer
    while iba <= ibEnd do
    begin
      // search end of Ispell output line
      // search end first because there may be the end of a line in the buffer
      while   ( ibb^ <> #10 )
          and ( ibb < ibEnd )
      do Inc (ibb);

      // copy section to work buffer
      StrLCopy (wbMark, iba, ibb - iba + 1);
      Inc (wbMark, ibb - iba + 1);

      // if line complete evaluate it
      if ibb^ = #10 then
      begin
        wbMark := workBuffer;

        // check if line is empty #13#10
        if workBuffer[1] = #10 then
        begin
          isEmptyLine := true;
        end
        else
        begin
          // Find misspelled word if none found yet.
          if result = false then
          begin
            misspelled := GetPosition (workBuffer, found);
            if misspelled = true then
            begin
              first := found;
              first.pos := found.pos - 1; // Ispell returns 1 for first position
                                          // Windows requires 0
              result := true;
             end;
          end;
        end;
      end; // if line complete

      // search for beginning of next line
      ibb := ibb + 1;
      iba := ibb;
    end; //while iba <= ibEnd
  end; //while not isEmptyLine
end;


end.

