/*
 * File: cycle.v256.c
 *   By: Dave Hiebeler
 *       hiebeler@turing.cs.rpi.edu
 *       Feb. 1990
 *
 * Griffeath's cyclic CA rule, described by A.K. Dewdney in his
 * "Computer Recreations" column in Scientific American, sometime in '89.
 *
 * parm1 is the number of states to use.
 *
 * This is the version which picks a random neighbor to compare itself to,
 * rather than checking all neighbors.
 *
 * An interesting starting-pattern is a square of randomness in a clear
 * background.
 * For example, on a 256x256 array, with parm1 set to 16, make a 40x40
 * square of cells with values between 0 and 15, out in the middle of
 * an otherwise empty array.
 */


#include "CMnborhood.h"

byte cycle_rule(), cycle_exit();
static CM_field_id_t direction, chosen_nbor;


void
init_function()
{

    update_function = cycle_rule;
    exit_function = cycle_exit;
    parm1 = 16;
    direction = CM_allocate_heap_field(2);
    chosen_nbor = CM_allocate_heap_field(8);
}


byte
cycle_rule()
{
    /* pick a random direction */
    CM_u_random_1L(direction, 2, 4);
    /* then get the appropriate neighbor to compare ourselves to */
    CM_u_eq_constant_1L(direction, 0, 2);
    CM_logand_context_with_test();
    CM_u_move_1L(chosen_nbor, VNorth, 8);
    CM_set_context();
    CM_u_eq_constant_1L(direction, 1, 2);
    CM_logand_context_with_test();
    CM_u_move_1L(chosen_nbor, VSouth, 8);
    CM_set_context();
    CM_u_eq_constant_1L(direction, 2, 2);
    CM_logand_context_with_test();
    CM_u_move_1L(chosen_nbor, VEast, 8);
    CM_set_context();
    CM_u_eq_constant_1L(direction, 3, 2);
    CM_logand_context_with_test();
    CM_u_move_1L(chosen_nbor, VWest, 8);
    
    CM_set_context();
    /* compare center+1 to chosen neighbor */
    CM_u_add_constant_2_1L(VCenter, 1, 8);
    CM_u_mod_constant_2_1L(VCenter, parm1, 8);
    CM_u_eq_1L(chosen_nbor, VCenter, 8);
    CM_logand_context_with_test();
    /* if equal, set cell to new value, the value of chosen_nbor.
     * otherwise, we won't change cell.
     */
    CM_u_move_1L(cell, chosen_nbor, 8);
    CM_set_context();
}


byte
cycle_exit()
{
    CM_deallocate_heap_field(direction);
    CM_deallocate_heap_field(chosen_nbor);
}
