/*
 *	Definitions for various arithmetic operations
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)xvecop.cc	2.2	9/18/89
 */

#define NO_VECTOR_MATHFUN
#include "rw/<T>Vec.h"
#define TYPE <T>_TYPE
#include "vecdefs.h"

static const char SCCSid[] = "@(#)xvecop.cc	2.2 9/18/89";

// Unary minus on a <T>Vec
<T>Vec
operator-(const <T>Vec& s)
{
  register i = s.length();
  <T>Vec temp(i);
  register <T>* sp = s.data();
  register <T>* dp = temp.data();
  register j = s.stride();
  while (i--) { *dp++ = -(*sp);  sp += j; }
  return temp;
}

#if HAS_INCRDECR
// Unary increment on a <T>Vec
<T>Vec&
<T>Vec::operator++()
{
  register i = length();
  register <T>* sp = data();
  register j = stride();
  while (i--) { ++(*sp); sp += j; }
  return *this;
}

// Unary decrement on a <T>Vec
<T>Vec&
<T>Vec::operator--()
{
  register i = length();
  register <T>* sp = data();
  register j = stride();
  while (i--) { --(*sp); sp += j; }
  return *this;
}

#endif

// Binary multiply element-by-element
<T>Vec
operator*(const <T>Vec& u, const <T>Vec& v)
{
  register i = v.length();
  u.lengthCheck(i);
  <T>Vec temp(i);
  register <T>* up = u.data();
  register <T>* vp = v.data();
  register <T>* dp = temp.data();
  register uj = u.stride();
  register vj = v.stride();
  while (i--) { *dp++ = *up * *vp;  up += uj; vp += vj; }
  return temp;
}

#if HAS_DIVIDE
// Binary divide element-by-element
<T>Vec
operator/(const <T>Vec& u, const <T>Vec& v)
{
  register i = v.length();
  u.lengthCheck(i);
  <T>Vec temp(i);
  register <T>* up = u.data();
  register <T>* vp = v.data();
  register <T>* dp = temp.data();
  register uj = u.stride();
  register vj = v.stride();
  while (i--) { *dp++ = *up / *vp;  up += uj; vp += vj; }
  return temp;
}
#endif

// Binary add element-by-element
<T>Vec
operator+(const <T>Vec& u, const <T>Vec& v)
{
  register i = v.length();
  u.lengthCheck(i);
  <T>Vec temp(i);
  register <T>* up = u.data();
  register <T>* vp = v.data();
  register <T>* dp = temp.data();
  register uj = u.stride();
  register vj = v.stride();
  while (i--) { *dp++ = *up + *vp;  up += uj; vp += vj; }
  return temp;
}

// Binary subtract element-by-element
<T>Vec
operator-(const <T>Vec& u, const <T>Vec& v)
{
  register i = v.length();
  u.lengthCheck(i);
  <T>Vec temp(i);
  register <T>* up = u.data();
  register <T>* vp = v.data();
  register <T>* dp = temp.data();
  register uj = u.stride();
  register vj = v.stride();
  while (i--) { *dp++ = *up - *vp;  up += uj; vp += vj; }
  return temp;
}

// Multiply by a scalar
<T>Vec
operator*(const <T>Vec& s, <T> scalar)
{
  register i = s.length();
  <T>Vec temp(i);
  register <T>* sp = s.data();
  register <T>* dp = temp.data();
  register j = s.stride();
  REGISTER <T> c = scalar;
  while (i--) { *dp++ = *sp * c; sp += j; }
  return temp;
}

#if HAS_DIVIDE
// Divide by a scalar
<T>Vec
operator/(const <T>Vec& s, <T> scalar)
{
  register i = s.length();
  <T>Vec temp(i);
  register <T>* sp = s.data();
  register <T>* dp = temp.data();
  register j = s.stride();
  REGISTER <T> c = scalar;
  while (i--) { *dp++ = *sp / c; sp += j; }
  return temp;
}

// Divide into a scalar
<T>Vec
operator/(<T> scalar, const <T>Vec& s)
{
  register i = s.length();
  <T>Vec temp(i);
  register <T>* sp = s.data();
  register <T>* dp = temp.data();
  register j = s.stride();
  REGISTER <T> c = scalar;
  while (i--) { *dp++ = c / *sp; sp += j; }
  return temp;
}
#endif

// Add a scalar
<T>Vec
operator+(const <T>Vec& s, <T> scalar)
{
  register i = s.length();
  <T>Vec temp(i);
  register <T>* sp = s.data();
  register <T>* dp = temp.data();
  register j = s.stride();
  REGISTER <T> c = scalar;
  while (i--) { *dp++ = *sp + c; sp += j; }
  return temp;
}

// Subtract a scalar
<T>Vec
operator-(const <T>Vec& s, <T> scalar)
{
  register i = s.length();
  <T>Vec temp(i);
  register <T>* sp = s.data();
  register <T>* dp = temp.data();
  register j = s.stride();
  REGISTER <T> c = scalar;
  while (i--) { *dp++ = *sp - c; sp += j; }
  return temp;
}

// Subtract from a scalar
<T>Vec
operator-(<T> scalar, const <T>Vec& s)
{
  register i = s.length();
  <T>Vec temp(i);
  register <T>* sp = s.data();
  register <T>* dp = temp.data();
  register j = s.stride();
  REGISTER <T> c = scalar;
  while (i--) { *dp++ = c - *sp; sp += j; }
  return temp;
}

/******** Arithmetic assignment operators **********/

<T>Vec&
<T>Vec::operator+=(const <T>Vec& u)
{
  register i = u.length();
  lengthCheck(i);
  register <T>* up = u.data();
  register <T>* tp = data();
  register uj = u.stride();
  register tj = stride();
  while (i--) { *tp += *up;  up += uj; tp += tj; }
  return *this;
}

<T>Vec&
<T>Vec::operator+=(<T> scalar)
{
  register i = length();
  register <T>* tp = data();
  register tj = stride();
  REGISTER <T> c = scalar;
  while (i--) { *tp += c; tp += tj; }
  return *this;
}

<T>Vec&
<T>Vec::operator-=(const <T>Vec& u)
{
  register i = u.length();
  lengthCheck(i);
  register <T>* up = u.data();
  register <T>* tp = data();
  register uj = u.stride();
  register tj = stride();
  while (i--) { *tp -= *up;  up += uj; tp += tj; }
  return *this;
}

<T>Vec&
<T>Vec::operator-=(<T> scalar)
{
  register i = length();
  register <T>* tp = data();
  register tj = stride();
  REGISTER <T> c = scalar;
  while (i--) { *tp -= c; tp += tj; }
  return *this;
}

<T>Vec&
<T>Vec::operator*=(const <T>Vec& u)
{
  register i = u.length();
  lengthCheck(i);
  register <T>* up = u.data();
  register <T>* tp = data();
  register uj = u.stride();
  register tj = stride();
  while (i--) { *tp *= *up;  up += uj; tp += tj; }
  return *this;
}

<T>Vec&
<T>Vec::operator*=(<T> scalar)
{
  register i = length();
  register <T>* tp = data();
  register tj = stride();
  REGISTER <T> c = scalar;
  while (i--) { *tp *= c; tp += tj; }
  return *this;
}

#if HAS_DIVIDE
<T>Vec&
<T>Vec::operator/=(const <T>Vec& u)
{
  register i = u.length();
  lengthCheck(i);
  register <T>* up = u.data();
  register <T>* tp = data();
  register uj = u.stride();
  register tj = stride();
  while (i--) { *tp /= *up;  up += uj; tp += tj; }
  return *this;
}

<T>Vec&
<T>Vec::operator/=(<T> scalar)
{
  register i = length();
  register <T>* tp = data();
  register tj = stride();
  REGISTER <T> c = scalar;
  while (i--) { *tp /= c; tp += tj; }
  return *this;
}

#endif
