 /*
  * Khoros: $Id: lviff2rast.c,v 1.1 1991/05/10 15:43:29 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: lviff2rast.c,v 1.1 1991/05/10 15:43:29 khoros Exp $";
#endif

 /*
  * $Log: lviff2rast.c,v $
 * Revision 1.1  1991/05/10  15:43:29  khoros
 * Initial revision
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lviff2rast.c
 >>>>
 >>>>      Program Name: viff2rast
 >>>>
 >>>> Date Last Updated: Sat Mar  2 14:47:37 1991 
 >>>>
 >>>>          Routines: lviff2rast - the library call for viff2rast
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
#include "file_formats/rast.h"
#include "vrev.h"
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lviff2rast - library call for viff2rast
*
* Purpose:
*    
*    Converts a VIFF image file to a Sun Raster image file.
*    
*    
* Input:
*    
*    Input to lviff2rast :
*    
*    viff_image     A pointer to struct xvimage, this image will  con-
*                   tain  the  image  to be converted from VIFF to SUN
*                   rast.
*    
*    invert_flag    resulting image is  either  photopositive  (1)  or
*                   photonegaative (0), if the type is BIT.
*    
*    
* Output:
*    
*    rast_image     A pointer to struct rast, this image will  contain
*                   the converted VIFF image.
*    
*    
*
* Written By: Tom Sauer
*    
*    
****************************************************************/


/* -library_def */
int     lviff2rast (viff_image, rast_image, invert_flag)

struct xvimage *viff_image;
struct rast **rast_image;
int invert_flag;

/* -library_def_end */

/* -library_code */
{

    int     width, height, depth, length, maptype, maplen;
    int     maprow, mapcol, i, count, num_times, num_bytes;
    unsigned char  *rast_ptr, *img_ptr, *r_ptr, *g_ptr, *b_ptr;
    struct rast *new_rast;

    char   *program = "lviff2rast";

    /* make sure the VIFF image is convertable */
    if ((!propertype (program, viff_image, VFF_TYP_BIT, FALSE)) &&
            (!propertype (program, viff_image, VFF_TYP_1_BYTE, FALSE))) {
        fprintf (stderr, "%s: VIFF image stored in an unconvertable data type.\n",
                program);
        fprintf (stderr, "Should be either bit or byte.\n");
        return (0);
    }
    if ((!proper_map_type (program, viff_image, VFF_MAPTYP_NONE, FALSE)) &&
            (!proper_map_type (program, viff_image, VFF_MAPTYP_1_BYTE, FALSE))) {
        fprintf (stderr, "%s: Map stored in unconvertable format.\n", program);
        fprintf (stderr, "Should be either VFF_MAPTYP_NONE or VFF_MAPTYP_1_BYTE.\n");
        return (0);
    }
    if ((!proper_num_bands (program, viff_image, 1, FALSE)) &&
            (!proper_num_bands (program, viff_image, 3, FALSE))) {
        fprintf (stderr, "%s: Image must have either 1 or 3 bands.\n", program);
        return (0);
    }
    if ((!proper_map_scheme (program, viff_image, VFF_MS_NONE, FALSE)) &&
            (!proper_map_scheme (program, viff_image, VFF_MS_ONEPERBAND, FALSE)) &&
            (!proper_map_scheme (program, viff_image, VFF_MS_SHARED, FALSE))) {
        fprintf (stderr, "%s: Unconvertable mapping scheme used.\n", program);
        fprintf (stderr, "Should be either VFF_MS_NONE, VFF_MS_ONEPERBAND or VFF_MS_SHARED.\n");
        return (0);
    }
    if (!proper_num_images (program, viff_image, 1, FALSE)) {
        fprintf (stderr, "%s: Can only convert files containing 1 image.\n",
                program);
        return (0);
    }
    if ((!proper_color_model (program, viff_image, VFF_CM_NONE, FALSE)) &&
            (!proper_color_model (program, viff_image, VFF_CM_genericRGB, FALSE)) &&
            (!proper_color_model (program, viff_image, VFF_CM_ntscRGB, FALSE)) &&
            (!proper_color_model (program, viff_image, VFF_CM_cieRGB, FALSE))) {
        fprintf (stderr, "%s: WARNING: Image uses non-RGB color space model.\n",
                program);
        fprintf (stderr, "Image will be interpreted as an RGB image.\n");
    }

    if ((!proper_map_type (program, viff_image, VFF_MAPTYP_NONE, FALSE)) &&
            (proper_num_bands (program, viff_image, 3, FALSE))) {
        fprintf (stderr, "%s: Can not convert an image with 3 bands and a map.\n",
                program);
        return (0);
    }

    /* set up variables to be used to construct the Raster file header */

    width   = viff_image->row_size;
    height  = viff_image->col_size;
    maprow  = viff_image->map_row_size;
    mapcol  = viff_image->map_col_size;
    maplen  = maprow * mapcol;
    maptype = (int) RMT_NONE;

         /* determine the correct depth for the rast image */

    switch (viff_image->data_storage_type) {
        case VFF_TYP_BIT: 
            depth = 1;
            num_bytes = ((width + 7) / 8);
            length = (num_bytes + num_bytes % 2) * height;
            break;
        case VFF_TYP_1_BYTE: 
            if (viff_image->num_data_bands == 1)
            {
                length = (width + width % 2) * height;
                depth = 8;
            }
            else
            {
                depth = 24;
                length = (3*width + (3*width) % 2) * height;
            }
            break;
    }

    /* allocate memory for the image header */

    new_rast = (struct rast *) calloc (1, sizeof (struct rast));
    if (new_rast == NULL)
    {
        fprintf (stderr, 
               "%s: Not enough memory available to create image header.\n", 
                program);
        return (0);
    }

    /* create the raster file header */

    new_rast->ras_magic = RAS_MAGIC;
    new_rast->ras_width = width;
    new_rast->ras_height = height;
    new_rast->ras_depth = depth;
    new_rast->ras_length = length;
    new_rast->ras_type = RT_STANDARD;
    new_rast->ras_maptype = maptype;

                      /* if map exists it is stored as bytes. If
                         there are not 3 columns in the map
                         there must be only 1 and therefore it
                         must be copied 3 times to make it into
                         an RGB map */
    if (maplen != 0) 
    {
        if (viff_image->map_row_size == 3) 
        {
            new_rast->ras_maptype = RMT_EQUAL_RGB;
            num_times = 1;      /* there are 3 columns in the map */
            new_rast->ras_maplength = maplen;
            /* allocate memory for the image */
            new_rast->data = (char *) malloc ((unsigned int) (length + maplen));
            if (new_rast->data == NULL)
            {
                fprintf (stderr, 
                 "%s: Not enough memory available to create image.\n", program);
                return (0);
            }
        }
        else 
        {
            new_rast->ras_maptype = RMT_EQUAL_RGB;
            num_times = 3;      /* there is only 1 column in the map */
            new_rast->ras_maplength = 3 * maplen;

            /* allocate memory for the image  and maps */

            new_rast->data = (char *) malloc (length + (3 * maplen));
            if (new_rast->data == NULL)
            {
                fprintf (stderr, 
                   "%s: Not enough memory available to create image.\n", 
                     program);
                return (0);
            }
        }
        
       /* copy the maps into the rast structure */

        rast_ptr = (unsigned char *) new_rast->data;
        for (count = 0; count < num_times; count++)
            bcopy (viff_image->maps, &rast_ptr[count * maplen], maplen);
        rast_ptr += num_times * maplen;
    }
    else if (viff_image->data_storage_type != VFF_TYP_BIT &&
             viff_image->num_data_bands == 1)
    {
       /* allocate memory for linear maps and the image */

       new_rast->ras_maptype = RMT_EQUAL_RGB;
       new_rast->ras_maplength = 3*256;
       new_rast->data = (char *) malloc ((unsigned int) (length + 3*256));
       if (new_rast->data == NULL)
       {
            fprintf (stderr, 
                   "%s: Not enough memory available to create image.\n", 
                    program);
            return (0);
       }

       /*
        *  since there is no maps, we need to make an implied
        *  linear (gray) map
        */
       rast_ptr = (unsigned char *) new_rast->data;
       for (count = 0; count < 256; count++)
       {
          rast_ptr[count] =
          rast_ptr[count + 256] =
          rast_ptr[count + 512] = (unsigned char) count;
       }
       rast_ptr = (unsigned char *) &new_rast->data[3*256];
    }
    else
    {
       new_rast->data = (char *) malloc ((unsigned int) (length));
       new_rast->ras_maplength = 0;
       if (new_rast->data == NULL)
       {
            fprintf (stderr, 
                   "%s: Not enough memory available to create image.\n", 
                    program);
            return (0);
       }
       rast_ptr = (unsigned char *) new_rast->data;
    }

    img_ptr = (unsigned char *) (viff_image->imagedata);

      /* copy the image data from viff to rast image */

    switch (depth) {
        case 1: 
            num_bytes = (width+7)/8;
            for (i = 0; i < num_bytes*height; i++)
            {
               if (invert_flag)
                  *rast_ptr = ~(rev[img_ptr[i]]);
               else
                  *rast_ptr = rev[img_ptr[i]];

               rast_ptr++;
               if (i % num_bytes == 0 && num_bytes % 2 == 1)
                  rast_ptr++;
            }
            break;
        case 8: 
            for (i = 0; i < width*height; i++)
            {
                *rast_ptr++ = *img_ptr++;
                if (i % width == 0 && width % 2 == 1)
                   rast_ptr++;
            }
            break;
        case 24: 
            r_ptr = (unsigned char *) (&viff_image->imagedata[0]);
            g_ptr = (unsigned char *) (&viff_image->imagedata[width * height]);
            b_ptr = (unsigned char *) (&viff_image->imagedata[2 * width * height]);

            /* transfer image that is stored in red-byte, green-byte,
               blue-byte order to all red followed by all green followed
               by all blue */

            for (i = 0; i < width*height; i++)
            {
                *rast_ptr++ = *r_ptr++;
                *rast_ptr++ = *g_ptr++;
                *rast_ptr++ = *b_ptr++;
                if (i % width == 0 && 3*width % 2 == 1)
                   rast_ptr++;
            }
            break;
    }

    *rast_image = new_rast;
    return (1);
}
/* -library_code_end */
